#pragma once

#include "police/argument_store.hpp"
#include "police/arguments.hpp"
#include "police/global_arguments.hpp"
#include "police/macros.hpp"
#include "police/parse_tree.hpp"
#include "police/storage/vector.hpp"
#include "police/utils/string.hpp"

#include <concepts>
#include <functional>
#include <limits>
#include <memory>
#include <optional>
#include <string_view>
#include <type_traits>
#include <typeindex>

namespace police {

class ArgumentsDefinition;
class OptionParser;

template <typename Factory, typename T>
concept FactoryFunction = requires(Factory f, const Arguments& opts) {
    { f(opts) } -> std::convertible_to<T>;
};

template <typename Definer>
concept ArgumentDefiner = requires(
    Definer d,
    ArgumentsDefinition& defs,
    const GlobalArguments& globals) { d(globals, defs); };

namespace details {
template <typename T>
struct is_shared_ptr_t : public std::false_type {};

template <typename T>
struct is_shared_ptr_t<std::shared_ptr<T>> : public std::true_type {};

} // namespace details

class ArgumentsDefinition {
    friend class OptionParser;

    struct Arg {
        virtual ~Arg() = default;

        virtual void parse(
            ArgumentStore& store,
            const std::string& key,
            const GlobalArguments& globals,
            const ParseTree& tree) = 0;

        virtual void set_default_value(
            ArgumentStore& store,
            const GlobalArguments& globals,
            const std::string& key) = 0;

        [[nodiscard]]
        virtual bool has_default_value() const = 0;

        [[nodiscard]]
        virtual std::string_view name() const = 0;

        virtual void
        validate(const GlobalArguments& globals, const ParseTree& tree)
            const = 0;
    };

    struct EnumArg final : public Arg {
        EnumArg(vector<std::string> values, const std::string& default_value);

        void parse(
            ArgumentStore& store,
            const std::string& key,
            const GlobalArguments& globals,
            const ParseTree& tree) override;

        void set_default_value(
            ArgumentStore& store,
            const GlobalArguments& globals,
            const std::string& key) override;

        [[nodiscard]]
        bool has_default_value() const override;

        [[nodiscard]]
        std::string_view name() const override;

        void validate(const GlobalArguments& globals, const ParseTree& tree)
            const override;

        vector<std::string> values_;
        int default_ = -1;
    };

    template <typename T>
    struct TArg final : public Arg {
        explicit TArg(std::optional<std::string> default_value)
            : default_value(default_value)
        {
        }

        void store_persistently(std::true_type, ArgumentStore& store, T obj)
        {
            store.store_persistently(std::move(obj));
        }

        void store_persistently(std::false_type, ArgumentStore&, const T&) {}

        void parse(
            ArgumentStore& store,
            const std::string& key,
            const GlobalArguments& globals,
            const ParseTree& tree) override;

        void set_default_value(
            ArgumentStore& store,
            const GlobalArguments& globals,
            const std::string& key) override
        {
            assert(has_default_value());
            parse(
                store,
                key,
                std::move(globals),
                ParseTree::parse(default_value.value()));
        }

        [[nodiscard]]
        bool has_default_value() const override
        {
            return default_value.has_value();
        }

        [[nodiscard]]
        std::string_view name() const override
        {
            return typeid(T).name();
        }

        void validate(const GlobalArguments& globals, const ParseTree& tree)
            const override;

        std::optional<std::string> default_value;
    };

public:
    template <typename T>
    void add_argument(
        const std::string& key,
        std::string_view description = "",
        std::optional<std::string> default_value = std::nullopt,
        bool optional = false)
    {
        if (std::count(keys_.begin(), keys_.end(), key)) {
            POLICE_INTERNAL_ERROR(
                "duplicate parameter with name " << key << " (type "
                                                 << typeid(T).name() << ")");
        }
        if (!std::is_same_v<T, std::string> && default_value.has_value() &&
            default_value.value().size() == 0u) {
            default_value = std::nullopt;
        } else if (
            std::is_same_v<T, std::string> && default_value.has_value() &&
            default_value.value().empty()) {
            default_value = "\"\"";
        }
        keys_.push_back(key);
        descriptions_.emplace_back(description);
        arg_types_.push_back(
            std::make_shared<TArg<T>>(std::move(default_value)));
        optional_.push_back(optional);
    }

    template <typename T>
    void add_ptr_argument(
        const std::string& key,
        std::string_view description = "",
        std::optional<std::string> default_value = std::nullopt,
        bool optional = false)
    {
        add_argument<std::shared_ptr<T>>(
            key,
            description,
            std::move(default_value),
            optional);
    }

    void add_enum_argument(
        const std::string& key,
        vector<std::string> values,
        std::string_view description = "",
        std::string default_value = "",
        bool optional = false)
    {

        if (std::count(keys_.begin(), keys_.end(), key)) {
            POLICE_INTERNAL_ERROR("duplicate parameter with name " << key);
        }
        keys_.push_back(key);
        descriptions_.emplace_back(description);
        arg_types_.push_back(
            std::make_shared<EnumArg>(
                std::move(values),
                std::move(default_value)));
        optional_.push_back(optional);
    }

    void dump(std::ostream& out, int shift = 0);

private:
    [[nodiscard]]
    ArgumentStore
    parse_arguments(const GlobalArguments& globals, const ParseTree& tree) const
    {
        ArgumentStore result;
        for (auto i = 0u; i < tree.args.size(); ++i) {
            arg_types_[i]->parse(result, keys_[i], globals, *tree.args[i]);
        }
        for (auto i = tree.args.size(); i < keys_.size(); ++i) {
            auto arg = tree.kwargs.find(keys_[i]);
            auto& arg_type = arg_types_[i];
            if (arg == tree.kwargs.end()) {
                if (optional_[i]) {
                    continue;
                }
                if (!arg_type->has_default_value()) {
                    POLICE_INVALID_ARGUMENT(
                        tree.raw,
                        "missing parameter " << keys_[i]);
                }
                arg_type->set_default_value(result, globals, keys_[i]);
            } else {
                arg_type->parse(result, keys_[i], globals, *arg->second);
            }
        }
        return result;
    }

    void
    validate_arguments(const GlobalArguments& globals, const ParseTree& tree)
        const
    {
        if (tree.args.size() > keys_.size()) {
            POLICE_INVALID_ARGUMENT(tree.raw, "invalid number of arguments");
        }
        for (auto it = tree.kwargs.begin(); it != tree.kwargs.end(); ++it) {
            if (std::count(keys_.begin(), keys_.end(), it->first) == 0u) {
                POLICE_INVALID_ARGUMENT(
                    tree.raw,
                    "unknown argument " << it->first << "=" << it->second->raw);
            }
        }
        for (auto i = 0u; i < tree.args.size(); ++i) {
            arg_types_[i]->validate(globals, *tree.args[i]);
        }
        for (auto i = tree.args.size(); i < keys_.size(); ++i) {
            auto arg = tree.kwargs.find(keys_[i]);
            auto& arg_type = arg_types_[i];
            if (arg == tree.kwargs.end()) {
                if (!optional_[i] && !arg_type->has_default_value()) {
                    POLICE_INVALID_ARGUMENT(
                        tree.raw,
                        "missing parameter " << keys_[i]);
                }
            } else {
                arg_type->validate(globals, *arg->second);
            }
        }
    }

private:
    vector<std::string> keys_;
    vector<std::string> descriptions_;
    vector<std::shared_ptr<Arg>> arg_types_;
    vector<bool> optional_;
};

class OptionParser {
    template <typename T>
    using parser_type = std::function<T(const Arguments&)>;

    using definer_type =
        std::function<void(const GlobalArguments&, ArgumentsDefinition&)>;

    struct OptionInfo {
        definer_type add_arguments;
        std::any factory;
    };

public:
    template <typename T, ArgumentDefiner Definer, FactoryFunction<T> Factory>
    static void
    add_option(std::string_view name, Factory&& factory, Definer&& args)
    {
        std::pair<std::type_index, std::string> key(typeid(T), name);
        OptionInfo infos(
            definer_type(std::forward<Definer>(args)),
            parser_type<T>(std::forward<Factory>(factory)));
        auto x = options_.emplace(std::move(key), std::move(infos));
        if (!x.second) {
            POLICE_INTERNAL_ERROR(
                "duplicate option " << name << " (type " << typeid(T).name()
                                    << ")" << std::endl);
        }
    }

    template <typename T>
    static void validate(std::string_view str)
    {
        validate<T>(std::make_shared<ParseTree>(ParseTree::parse(str)));
    }

    template <typename T>
    static void validate(const GlobalArguments& globals, const ParseTree& tree)
    {
        if constexpr (std::is_same_v<T, std::string>) {
            if (!tree.is_leaf()) {
                POLICE_INVALID_ARGUMENT(tree.raw, "expected string");
            }
        } else if constexpr (std::is_same_v<T, int_t>) {
            if (!tree.is_leaf()) {
                POLICE_INVALID_ARGUMENT(tree.raw, "expected integer");
            }
            if (tree.raw == "-inf" || tree.raw == "-infinity" ||
                tree.raw == "inf" || tree.raw == "infinity") {
                return;
            }
            try {
                std::stoi(tree.raw);
            } catch (std::invalid_argument&) {
                POLICE_INVALID_ARGUMENT(tree.raw, "not an integer");
            } catch (std::out_of_range&) {
                POLICE_INVALID_ARGUMENT(tree.raw, "out of range");
            }
        } else if constexpr (std::is_same_v<T, real_t>) {
            if (!tree.is_leaf()) {
                POLICE_INVALID_ARGUMENT(tree.raw, "expected number");
            }
            if (tree.raw == "-inf" || tree.raw == "-infinity" ||
                tree.raw == "inf" || tree.raw == "infinity") {
                return;
            }
            try {
                std::stod(tree.raw);
            } catch (std::invalid_argument&) {
                POLICE_INVALID_ARGUMENT(tree.raw, "not a number");
            } catch (std::out_of_range&) {
                POLICE_INVALID_ARGUMENT(tree.raw, "out of range");
            }
        } else if constexpr (std::is_same_v<T, bool>) {
            if (!tree.is_leaf()) {
                POLICE_INVALID_ARGUMENT(tree.raw, "expected Boolean");
            }
            const auto argl = std::string(police::tolower(tree.raw));
            if (argl == "true") {
            } else if (argl == "false") {
            } else {
                try {
                    std::stoi(argl);
                } catch (std::exception&) {
                    POLICE_INVALID_ARGUMENT(tree.raw, "not a Boolean");
                }
            }
        } else {
            auto pos = options_.find(
                std::pair<std::type_index, std::string>(typeid(T), tree.name));
            if (pos == options_.end()) {
                POLICE_INVALID_ARGUMENT(
                    tree.raw,
                    "no matching " << typeid(T).name() << " option with name "
                                   << tree.name);
            }
            auto& info = pos->second;
            ArgumentsDefinition arg_parser;
            info.add_arguments(globals, arg_parser);
            arg_parser.validate_arguments(globals, tree);
        }
    }

    template <typename T>
    static T parse(const GlobalArguments& globals, const ParseTree& tree)
    {
        validate<T>(globals, tree);
        if constexpr (std::is_same_v<T, std::string>) {
            if (!tree.is_leaf()) {
                POLICE_INVALID_ARGUMENT(tree.raw, "expected string");
            }
            return tree.raw;
        } else if constexpr (std::is_same_v<T, int_t>) {
            if (!tree.is_leaf()) {
                POLICE_INVALID_ARGUMENT(tree.raw, "expected integer");
            }
            if (tree.raw == "-inf" || tree.raw == "-infinity") {
                return std::numeric_limits<int_t>::min();
            }
            if (tree.raw == "inf" || tree.raw == "infinity") {
                return std::numeric_limits<int_t>::max();
            }
            try {
                return std::stoi(tree.raw);
            } catch (std::invalid_argument&) {
                POLICE_INVALID_ARGUMENT(tree.raw, "not an integer");
            } catch (std::out_of_range&) {
                POLICE_INVALID_ARGUMENT(tree.raw, "out of range");
            }
        } else if constexpr (std::is_same_v<T, real_t>) {
            if (!tree.is_leaf()) {
                POLICE_INVALID_ARGUMENT(tree.raw, "expected number");
            }
            if (tree.raw == "-inf" || tree.raw == "-infinity") {
                return -std::numeric_limits<int_t>::infinity();
            }
            if (tree.raw == "inf" || tree.raw == "infinity") {
                return std::numeric_limits<int_t>::infinity();
            }
            try {
                return std::stod(tree.raw);
            } catch (std::invalid_argument&) {
                POLICE_INVALID_ARGUMENT(tree.raw, "not a number");
            } catch (std::out_of_range&) {
                POLICE_INVALID_ARGUMENT(tree.raw, "out of range");
            }
        } else if constexpr (std::is_same_v<T, bool>) {
            if (!tree.is_leaf()) {
                POLICE_INVALID_ARGUMENT(tree.raw, "expected Boolean");
            }
            const auto argl = std::string(police::tolower(tree.raw));
            if (argl == "true") {
                return true;
            } else if (argl == "false") {
                return false;
            } else {
                try {
                    return std::stoi(argl);
                } catch (std::exception&) {
                    POLICE_INVALID_ARGUMENT(tree.raw, "not a Boolean");
                }
            }
        } else {
            auto pos = options_.find(
                std::pair<std::type_index, std::string>(typeid(T), tree.name));
            if (pos == options_.end()) {
                POLICE_INVALID_ARGUMENT(
                    tree.raw,
                    "no matching " << typeid(T).name() << " option with name "
                                   << tree.name);
            }
            auto& info = pos->second;
            ArgumentsDefinition arg_parser;
            info.add_arguments(globals, arg_parser);
            Arguments args(globals, arg_parser.parse_arguments(globals, tree));
            auto factory = std::any_cast<parser_type<T>>(info.factory);
            return factory(args);
        }
    }

    template <typename T>
    static T parse(const GlobalArguments& globals, std::string_view str)
    {
        return parse<T>(globals, ParseTree::parse(str));
    }

    static void dump(const GlobalArguments& globals, std::ostream& out);

private:
    static unordered_map<std::pair<std::type_index, std::string>, OptionInfo>
        options_;
};

template <typename T>
void ArgumentsDefinition::TArg<T>::parse(
    ArgumentStore& store,
    const std::string& key,
    const GlobalArguments& globals,
    const ParseTree& tree)
{
    auto obj = OptionParser::parse<T>(globals, tree);
    store_persistently(details::is_shared_ptr_t<T>(), *globals.storage, obj);
    store.set<T>(key, std::move(obj));
}

template <typename T>
void ArgumentsDefinition::TArg<T>::validate(
    const GlobalArguments& globals,
    const ParseTree& tree) const
{
    OptionParser::validate<T>(globals, tree);
}

} // namespace police
