#pragma once

#include "police/base_types.hpp"
#include "police/expressions/expression.hpp"
#include "police/expressions/expression_evaluator.hpp"
#include "police/macros.hpp"
#include "police/verification_property.hpp"

namespace police {

template <typename SuccessorGenerator>
class PlanValidator {
public:
    enum Status {
        VALID,
        EMPTY_SEQUENCE,
        INVALID_START,
        INVALID_SUCCESSOR,
        SATISFIED_REACH,
        INVALID_AVOID,
    };

    struct Result {
        size_t index;
        Status status;
    };

    explicit PlanValidator(
        SuccessorGenerator successor_generator,
        const VerificationProperty* property)
        : succ_gen_(std::move(successor_generator))
        , property_(std::move(property))
    {
    }

    [[nodiscard]]
    static std::string_view status_to_string(Status status)
    {
        switch (status) {
        case (VALID): return "valid";
        case EMPTY_SEQUENCE: return "empty sequence";
        case INVALID_START: return "invalid start state";
        case INVALID_SUCCESSOR: return "invalid successor state";
        case SATISFIED_REACH: return "goal condition satisfied";
        case INVALID_AVOID: return "avoid condition violated";
        }
        POLICE_UNREACHABLE();
    }

    template <typename Range>
    Result validate_state_sequence(Range&& range)
    {
        return validate_state_sequence(range.begin(), range.end());
    }

    template <typename Iterator, typename Sentinel>
    Result validate_state_sequence(Iterator begin, Sentinel end)
    {
        if (begin == end) {
            return {0, EMPTY_SEQUENCE};
        }
        if (!check_condition(property_->start, *begin)) {
            return {0, INVALID_START};
        }
        if (check_condition(property_->reach, *begin)) {
            return {0, SATISFIED_REACH};
        }
        size_t step = 0;
        auto succ = begin + 1;
        for (; succ != end; ++begin, ++succ, ++step) {
            if (!check_successor(*begin, *succ)) {
                return {step, INVALID_SUCCESSOR};
            }
            if (check_condition(property_->reach, *succ)) {
                return {step + 1, SATISFIED_REACH};
            }
        }
        if (!check_condition(property_->avoid, *begin)) {
            return {step, INVALID_AVOID};
        }
        return {step, VALID};
    }

private:
    template <typename State>
    static bool
    check_condition(const expressions::Expression& expr, State&& state)
    {
        auto get_value = [&state](size_t var) { return (state[var]); };
        return static_cast<bool>(expressions::evaluate(expr, get_value));
    }

    template <typename State>
    bool check_successor(State&& state, State&& successor)
    {
        auto successors = succ_gen_(std::forward<State>(state));
        for (const auto& [succ, _] : successors) {
            if (std::equal(succ.begin(), succ.end(), successor.begin())) {
                return true;
            }
        }
        return false;
    }

    SuccessorGenerator succ_gen_;
    const VerificationProperty* property_;
};

} // namespace police
