#pragma once

#include "police/expressions/expression.hpp"
#include "police/sat_model.hpp"
#include "police/storage/variable_space.hpp"
#include "police/storage/vector.hpp"

#include <string_view>

namespace police {

namespace expressions {
class Expression;
}

class SMT {
public:
    using model_type = SATModel;
    enum class Status { SAT, UNSAT };

    SMT() = default;
    SMT(const SMT&) = delete;
    virtual ~SMT() = default;

    void push_snapshot();
    void pop_snapshot();
    void clear();

    size_t add_variables(const VariableSpace& vspace);
    size_t add_variable(std::string_view name, VariableType type);

    void add_constraint(const expressions::Expression& expr);

    [[nodiscard]]
    virtual Status solve() = 0;

    [[nodiscard]]
    virtual Status
    solve(const vector<expressions::Expression>& assumptions) = 0;

    [[nodiscard]]
    virtual model_type get_model() const = 0;

    [[nodiscard]]
    virtual vector<expressions::Expression> get_unsat_core() const = 0;

    virtual void dump(std::ostream&) const {}

    [[nodiscard]]
    size_t num_variables() const;

    void mirror(SMT& other) const;

protected:
    [[nodiscard]]
    const VariableSpace& get_variables() const
    {
        return vars_;
    }

    [[nodiscard]]
    const vector<expressions::Expression>& get_constraints() const
    {
        return constraints_;
    }

    // SMT& operator=(const)

private:
    virtual void do_push_snapshot() = 0;
    virtual void do_pop_snapshot() = 0;
    virtual void do_clear() = 0;
    virtual size_t
    do_add_variable(std::string_view name, VariableType type) = 0;
    virtual void do_add_constraint(const expressions::Expression& expr) = 0;

    struct Snapshot {
        size_t num_vars;
        size_t num_constraints;
    };

    vector<Snapshot> snapshots_;
    VariableSpace vars_;
    vector<expressions::Expression> constraints_;
};

} // namespace police
