#pragma once

#include "police/storage/vector.hpp"

#include <algorithm>
#include <cassert>
#include <cstdint>

namespace police {

template <typename _T = std::uint32_t>
class index_set {
public:
    using size_type = std::uint32_t;
    using index_type = _T;
    using set_type = vector<index_type>;

    explicit index_set(size_type num_indices = 0)
        : index_to_set_(num_indices)
    {
    }

    [[nodiscard]]
    size_type size() const
    {
        return size_;
    }

    [[nodiscard]]
    bool empty() const
    {
        return size() == 0;
    }

    index_type insert(const set_type& set)
    {
        const auto id = set_sizes_.size();
        for (const auto& index : set) {
            assert(index < index_to_set_.size());
            index_to_set_[index].push_back(id);
        }
        set_sizes_.push_back(set.size());
        hit_.push_back(0);
        if (set.empty()) {
            empty_sets_.push_back(id);
        }
        return id;
    }

    template <typename Callback>
    void forall_subsets(const set_type& set, Callback callback) const
    {
        std::copy(set_sizes_.begin(), set_sizes_.end(), hit_.begin());
        for (const auto& idx : set) {
            assert(idx < index_to_set_.size());
            for (const auto& subs : index_to_set_[idx]) {
                if (--hit_[subs] == 0) {
                    callback(subs);
                }
            }
        }
    }

    template <typename Callback>
    void forall_supersets(const set_type& set, Callback callback) const
    {
        std::fill(hit_.begin(), hit_.end(), set.size());
        for (const auto& idx : set) {
            assert(idx < index_to_set_.size());
            for (const auto& subs : index_to_set_[idx]) {
                if (--hit_[subs] == 0) {
                    callback(subs);
                }
            }
        }
    }

private:
    vector<vector<index_type>> index_to_set_;
    vector<size_type> set_sizes_;
    mutable vector<size_type> hit_;
    vector<index_type> empty_sets_;
    size_type size_ = 0;
};

} // namespace police
