#pragma once

#include "police/storage/segmented_vector.hpp"

#include <array>
#include <cassert>
#include <cstdint>
#include <cstring>

namespace police {

template <typename Bin = std::uint32_t, std::size_t PageSize = 1u << 16u>
struct segmented_arrays {
    using page_t = std::array<Bin, PageSize>;
    using size_t = std::size_t;

    constexpr explicit segmented_arrays(size_t array_length)
        : array_length_(array_length)
        , per_page_(PageSize / array_length)
        , page_offset_(per_page_)
    {
        assert(array_length_ <= PageSize);
    }

    constexpr size_t allocate()
    {
        if (!free_slots_.empty()) {
            const auto pos = free_slots_.back();
            free_slots_.pop_back();
            return pos;
        }
        if (page_offset_ >= per_page_) {
            pages_.emplace_back();
            pages_.back().fill(0);
            page_offset_ = 0;
        }
        return (pages_.size() - 1) * per_page_ + page_offset_++;
    }

    constexpr void deallocate(std::size_t index)
    {
        free_slots_.push_back(index);
        Bin* bin = operator[](index);
        std::fill(bin, bin + array_length_, 0);
    }

    constexpr Bin* operator[](size_t pos)
    {
        const auto [page, off] = get_coordinate(pos);
        return pages_[page].data() + off * array_length_;
    }

    constexpr const Bin* operator[](size_t pos) const
    {
        const auto [page, off] = get_coordinate(pos);
        return pages_[page].data() + off * array_length_;
    }

    constexpr Bin* at(size_t pos) { return operator[](pos); }

    constexpr const Bin* at(size_t pos) const { return operator[](pos); }

    constexpr size_t size() const
    {
        return pages_.empty() ? 0ul
                              : (pages_.size() - 1) * per_page_ + page_offset_ -
                                    free_slots_.size();
    }

private:
    constexpr size_t get_index(size_t page_index, size_t page_offset) const
    {
        return page_index * per_page_ + page_offset;
    }

    constexpr std::pair<size_t, size_t> get_coordinate(size_t index) const
    {
        const size_t page_index = index / per_page_;
        return {page_index, index - page_index * per_page_};
    }

    segmented_vector<page_t> pages_;
    segmented_vector<size_t> free_slots_;
    size_t array_length_ = 0;
    size_t per_page_ = 0;
    size_t page_offset_ = 0;
};

} // namespace police
