#pragma once

#include "police/base_types.hpp"
#include "police/macros.hpp"
#include "police/utils/hash.hpp"
#include "police/utils/type_traits.hpp"

#include <functional>
#include <type_traits>
#include <variant>

namespace police {

class Value {
    template <typename T>
    friend struct police::hash;

public:
    enum class Type {
        BOOL,
        INT,
        REAL,
    };

    template <typename T>
    constexpr Value(T&& value, const Type type)
        : value_(0)
    {
        switch (type) {
        case Type::BOOL: value_ = static_cast<bool>(value); break;
        case Type::INT: value_ = static_cast<int_t>(value); break;
        case Type::REAL: value_ = static_cast<real_t>(value); break;
        }
    }

    constexpr explicit Value(bool value)
        : value_(value)
    {
    }

    constexpr explicit Value(int_t value)
        : value_(value)
    {
    }

    constexpr explicit Value(real_t value)
        : value_(value)
    {
    }

    constexpr explicit Value(Type value)
        : value_(0)
    {
        switch (value) {
        case Type::BOOL: value_ = false; break;
        case Type::INT: value_ = 0; break;
        case Type::REAL: value_ = static_cast<real_t>(0.); break;
        }
    }

    constexpr operator bool() const { return cast<Type::BOOL>(value_); }

    constexpr operator int_t() const { return cast<Type::INT>(value_); }

    constexpr operator real_t() const { return cast<Type::REAL>(value_); }

    constexpr Value& operator=(bool val)
    {
        value_ = val;
        return *this;
    }

    constexpr Value& operator=(int_t val)
    {

        value_ = val;
        return *this;
    }

    constexpr Value& operator=(real_t val)
    {

        value_ = val;
        return *this;
    }

    constexpr Value operator&&(const Value& other) const
    {
        return Value(static_cast<bool>(*this) && static_cast<bool>(other));
    }

    constexpr Value operator||(const Value& other) const
    {
        return Value(static_cast<bool>(*this) || static_cast<bool>(other));
    }

    constexpr Value operator!() const
    {
        return Value(!static_cast<bool>(*this));
    }

    constexpr Value operator+(const Value& other) const
    {
        return combine(value_, other.value_, std::plus{});
    }

    constexpr Value operator-(const Value& other) const
    {
        return combine(value_, other.value_, std::minus{});
    }

    constexpr Value operator*(const Value& other) const
    {
        return combine(value_, other.value_, std::multiplies{});
    }

    constexpr Value operator/(const Value& other) const
    {
        return Value(static_cast<real_t>(*this) / static_cast<real_t>(other));
    }

    constexpr Value operator%(const Value& other) const
    {
        return Value{cast<Type::INT>(value_) % cast<Type::INT>(other.value_)};
    }

    constexpr bool operator==(const Value& other) const
    {
        return cast<Type::BOOL>(combine(value_, other.value_, std::equal_to{}));
    }

    constexpr bool operator<=(const Value& other) const
    {

        return cast<Type::BOOL>(
            combine(value_, other.value_, std::less_equal{}));
    }

    constexpr bool operator>=(const Value& other) const
    {
        return other <= *this;
    }

    constexpr bool operator>(const Value& other) const
    {
        return !(*this <= other);
    }

    constexpr bool operator<(const Value& other) const
    {
        return !(other <= *this);
    }

    constexpr Type get_type() const { return get_type(value_); }

    constexpr void swap(Value& other) { std::swap(value_, other.value_); }

    friend std::ostream& operator<<(std::ostream& out, const Value& val)
    {
        std::visit(
            [&](auto&& val) -> std::ostream& { return (out << val); },
            val.value_);
        switch (val.get_type()) {
        case Value::Type::INT: return out << 'i';
        case Value::Type::REAL: return out << 'r';
        case Value::Type::BOOL: return out << 'b';
        }
        POLICE_UNREACHABLE();
    }

private:
    using ValueData = std::variant<bool, int_t, real_t>;

    constexpr Value(ValueData&& val)
        : value_(std::move(val))
    {
    }

    constexpr static Type meet(Type t0, Type t1)
    {
        if (t0 == Type::REAL || t1 == Type::REAL) {
            return Type::REAL;
        } else if (t0 == Type::INT || t1 == Type::INT) {
            return Type::INT;
        } else {
            return Type::BOOL;
        }
    }

    template <Type t>
    using return_type_t =
        ite_t<t == Type::BOOL, bool, ite_t<t == Type::INT, int_t, real_t>>;

    template <Type t>
    constexpr static return_type_t<t> cast(ValueData val)
    {
        return std::visit([](auto&& x) -> return_type_t<t> { return x; }, val);
    }

    constexpr static Type get_type(ValueData val)
    {
        return std::visit(
            [](auto&& x) {
                using T = std::decay_t<decltype(x)>;
                if constexpr (std::is_same_v<T, bool>) {
                    return Type::BOOL;
                } else if constexpr (std::is_same_v<T, int_t>) {
                    return Type::INT;
                } else {
                    return Type::REAL;
                }
            },
            val);
    }

    template <typename Op>
    constexpr static ValueData
    combine(const ValueData& v0, const ValueData& v1, Op&& op)
    {
        switch (meet(get_type(v0), get_type(v1))) {
        case Type::BOOL:
            return {op(cast<Type::BOOL>(v0), cast<Type::BOOL>(v1))};
        case Type::INT: return {op(cast<Type::INT>(v0), cast<Type::INT>(v1))};
        case Type::REAL:
            return {op(cast<Type::REAL>(v0), cast<Type::REAL>(v1))};
        }
        POLICE_UNREACHABLE();
    }

    ValueData value_;
};

#define DEFINE_CMP_OPS(Type)                                                   \
    constexpr inline bool operator==(Value a, Type b)                          \
    {                                                                          \
        return a == Value(b);                                                  \
    }                                                                          \
                                                                               \
    constexpr inline bool operator!=(Value a, Type b)                          \
    {                                                                          \
        return a != Value(b);                                                  \
    }                                                                          \
                                                                               \
    constexpr inline bool operator<=(Value a, Type b)                          \
    {                                                                          \
        return a <= Value(b);                                                  \
    }                                                                          \
                                                                               \
    constexpr inline bool operator<(Value a, Type b)                           \
    {                                                                          \
        return a < Value(b);                                                   \
    }                                                                          \
                                                                               \
    constexpr inline bool operator>=(Value a, Type b)                          \
    {                                                                          \
        return a >= Value(b);                                                  \
    }                                                                          \
                                                                               \
    constexpr inline bool operator>(Value a, Type b)                           \
    {                                                                          \
        return a > Value(b);                                                   \
    }                                                                          \
                                                                               \
    constexpr inline bool operator==(Type b, Value a)                          \
    {                                                                          \
        return a == Value(b);                                                  \
    }                                                                          \
                                                                               \
    constexpr inline bool operator!=(Type b, Value a)                          \
    {                                                                          \
        return a != Value(b);                                                  \
    }                                                                          \
                                                                               \
    constexpr inline bool operator<=(Type b, Value a)                          \
    {                                                                          \
        return a >= Value(b);                                                  \
    }                                                                          \
                                                                               \
    constexpr inline bool operator<(Type b, Value a)                           \
    {                                                                          \
        return a > Value(b);                                                   \
    }                                                                          \
                                                                               \
    constexpr inline bool operator>=(Type b, Value a)                          \
    {                                                                          \
        return a <= Value(b);                                                  \
    }                                                                          \
                                                                               \
    constexpr inline bool operator>(Type b, Value a)                           \
    {                                                                          \
        return a < Value(b);                                                   \
    }

DEFINE_CMP_OPS(real_t)
DEFINE_CMP_OPS(int_t)
DEFINE_CMP_OPS(bool)

#undef DEFINE_CMP_OPS

template <>
struct hash<Value> {
    std::size_t operator()(const Value& value) const
    {
        return std::hash<Value::ValueData>()(value.value_);
    }
};

} // namespace police
