#pragma once

#include "police/base_types.hpp"
#include "police/storage/value.hpp"
#include "police/storage/vector.hpp"

#include <algorithm>
#include <iterator>

namespace police {

class ValueVector {
public:
    using iterator = vector<Value>::iterator;
    using const_iterator = vector<Value>::const_iterator;
    using value_type = Value;
    using reference_type = value_type&;
    using const_reference_type = const value_type&;

    template <typename Iterator>
    constexpr ValueVector(Iterator begin, Iterator end)
        : values_(begin, end)
    {
    }

    constexpr explicit ValueVector(size_t size)
        : values_(size, Value(false))
    {
    }

    constexpr ValueVector() = default;

    [[nodiscard]]
    constexpr iterator begin()
    {
        return values_.begin();
    }

    [[nodiscard]]
    constexpr iterator end()
    {
        return values_.end();
    }

    [[nodiscard]]
    constexpr const_iterator begin() const
    {
        return values_.begin();
    }

    [[nodiscard]]
    constexpr const_iterator end() const
    {
        return values_.end();
    }

    [[nodiscard]]
    constexpr const_iterator cbegin() const
    {
        return values_.begin();
    }

    [[nodiscard]]
    constexpr const_iterator cend() const
    {
        return values_.end();
    }

    [[nodiscard]]
    constexpr size_t size() const
    {
        return values_.size();
    }

    [[nodiscard]]
    constexpr const_reference_type operator[](size_t idx) const
    {
        return values_[idx];
    }

    [[nodiscard]]
    constexpr reference_type operator[](size_t idx)
    {
        return values_[idx];
    }

    template <typename T>
    [[nodiscard]]
    constexpr vector<T> cast_as() const
    {
        vector<T> result;
        result.reserve(values_.size());
        std::transform(
            values_.begin(),
            values_.end(),
            std::back_inserter(result),
            [](auto&& val) { return static_cast<T>(val); });
        return result;
    }

    void push_back(Value value) { values_.push_back(std::move(value)); }

private:
    vector<Value> values_;
};

} // namespace police
