#pragma once

#include "police/base_types.hpp"
#include "police/storage/vector.hpp"

#include <cassert>
#include <ranges>

namespace police {

namespace detail {
// https://www.reddit.com/r/cpp_questions/comments/15xf67b/how_to_define_a_concept_for_a_nested_range_with/
template <typename R, template <typename> class Element_Trait>
concept nested_range_satisfies = Element_Trait<
    std::ranges::range_value_t<std::ranges::range_value_t<R>>>::value;

template <typename R>
concept nested_unsigned_range =
    std::ranges::range<R> and nested_range_satisfies<R, std::is_unsigned>;

} // namespace detail

class DependencyQueue {
public:
    template <detail::nested_unsigned_range Dependencies>
    explicit DependencyQueue(Dependencies&& deps)
    {
        police::size_t i = 0;
        for (const auto& i_deps : deps) {
            police::size_t j = 0;
            for (const auto& dep : i_deps) {
                if (dep >= deps_.size()) {
                    deps_.resize(dep + 1);
                }
                deps_[dep].push_back(i);
                ++j;
            }
            if (j == 0) {
                queue_.push_back(i);
            }
            open_deps_.push_back(j);
            ++i;
        }
        assert(i >= deps_.size());
        deps_.resize(i);
    }

    [[nodiscard]]
    constexpr bool empty() const
    {
        return popped_ == deps_.size();
    }

    [[nodiscard]]
    constexpr bool can_pop() const
    {
        return !queue_.empty();
    }

    [[nodiscard]]
    constexpr police::size_t pop()
    {
        assert(can_pop());
        const police::size_t elem = queue_.back();
        queue_.pop_back();
        ++popped_;
        for (const auto& dep : deps_[elem]) {
            assert(open_deps_[dep] > 0);
            if (--open_deps_[dep] == 0) {
                queue_.push_back(dep);
            }
        }
        return elem;
    }

private:
    vector<vector<police::size_t>> deps_;
    vector<police::size_t> open_deps_;
    vector<police::size_t> queue_;
    police::size_t popped_ = 0;
};

} // namespace police
