#pragma once

#include <chrono>
#include <ostream>
#include <string>
#include <string_view>

namespace police {

class Chronometer {
public:
    using clock_type = std::chrono::high_resolution_clock;
    using duration_type = clock_type::duration;
    using time_stamp_type = clock_type::time_point;

    Chronometer();

    [[nodiscard]]
    duration_type operator()() const;

private:
    time_stamp_type tstmp_;
};

class StopWatch {
public:
    enum Unit {
        SILENT,
        MILLISECONDS,
        SECONDS,
    };

    class ScopedTimer {
    public:
        explicit ScopedTimer(StopWatch* watch);
        ~ScopedTimer();

    private:
        Chronometer clock_;
        StopWatch* watch_;
    };

    explicit StopWatch(bool start = true);

    [[nodiscard]]
    Chronometer::duration_type get_duration() const;

    [[nodiscard]]
    double get_milliseconds() const;

    [[nodiscard]]
    double get_seconds() const;

    void reset();

    void resume();

    void stop();

    [[nodiscard]]
    ScopedTimer scope();

    template <typename Fn, typename... Args>
    auto function_call(Fn fn, Args&&... args)
    {
        auto t = scope();
        return fn(std::forward<Args>(args)...);
    }

    StopWatch& operator+=(const StopWatch& clock);

private:
    friend class ScopedTimer;

    Chronometer::duration_type dur_;
    Chronometer clock_;
    bool running_ = false;
};

class ScopedStopWatch : public StopWatch {
public:
    enum Unit {
        MILLISECONDS,
        SECONDS,
    };

    explicit ScopedStopWatch(
        std::string_view name,
        bool start = true,
        Unit unit = MILLISECONDS);

    ~ScopedStopWatch();

    void cancel();

    void destroy();

    ScopedStopWatch& operator+=(const StopWatch& clock);

private:
    const std::string name_;
    const Unit unit_;
    bool destroyed_ = false;
};

std::ostream& operator<<(std::ostream& out, const StopWatch& watch);

} // namespace police
