#pragma once

#include "police/base_types.hpp"
#include "police/storage/flat_state.hpp"
#include "police/storage/id_map.hpp"
#include "police/storage/segmented_vector.hpp"
#include "police/storage/vector.hpp"
#include "police/verifiers/ic3/cube.hpp"
#include <cstdint>
#include <limits>
#include <memory>

namespace police::ic3 {

class CubeDatabase {
private:
    struct SubsumptionCache {
        constexpr static std::uint8_t NOT_CACHED = 0;
        constexpr static std::uint8_t I_SUBSUMES_J = 1;
        constexpr static std::uint8_t J_SUBSUMES_I = 2;
        constexpr static std::uint8_t INCOMPARABLE = 3;

        explicit SubsumptionCache(const id_map<Cube>::container_type* cubes);

        [[nodiscard]]
        bool operator()(size_t i, size_t j) const;

        void scrub(const vector<size_t>& idxs);

        mutable vector<vector<std::uint8_t>> cache;
        const id_map<Cube>::container_type* cubes;
    };

public:
    using iterator = id_map<Cube>::iterator;

    constexpr static size_t NO_FRAME = std::numeric_limits<size_t>::max();

    constexpr static size_t INF_FRAME = NO_FRAME - 1;

    CubeDatabase();

    void clear();

    [[nodiscard]]
    iterator begin() const;

    [[nodiscard]]
    iterator end() const;

    [[nodiscard]]
    size_t size() const;

    std::pair<iterator, bool> insert(Cube cube);

    void remove(size_t cube_idx);

    [[nodiscard]]
    const Cube& get_cube(size_t cube_idx) const;

    [[nodiscard]]
    size_t get_frame(size_t cube_idx) const;

    void set_frame(size_t cube_idx, size_t frame_idx);

    [[nodiscard]]
    bool subsumes(size_t idx1, size_t idx2) const;

    [[nodiscard]]
    size_t num_dangling_cubes() const
    {
        return dangling_cubes_.size();
    }

    vector<size_t> collect_garbage();

private:
    void filter_no_frame_cubes();
    void scrub_frame_ids(const vector<size_t>& cubes);

    std::unique_ptr<id_map<Cube>> cubes_;
    vector<size_t> frame_idx_;
    vector<size_t> dangling_cubes_;
    SubsumptionCache subsumes_;
};

class Frame {
public:
    using const_iterator = vector<size_t>::const_iterator;
    using iterator = const_iterator;

    explicit Frame(CubeDatabase* db, size_t frame_idx);

    [[nodiscard]]
    size_t index() const
    {
        return frame_idx_;
    }

    [[nodiscard]]
    size_t operator[](size_t idx) const
    {
        return cubes_[idx];
    }

    [[nodiscard]]
    const_iterator begin() const
    {
        return cubes_.begin();
    }

    [[nodiscard]]
    const_iterator end() const
    {
        return cubes_.end();
    }

    [[nodiscard]]
    bool empty() const
    {
        return cubes_.empty();
    }

    [[nodiscard]]
    size_t size() const
    {
        return cubes_.size();
    }

    void clear() { cubes_.clear(); }

    iterator insert(size_t cube_id);

    iterator erase(iterator pos);

    void remove_if_subsuming(size_t cube_id);

    void remove(size_t cube_id);

    [[nodiscard]]
    const_iterator find(size_t cube_id) const;

    [[nodiscard]]
    bool contains_subsumed(size_t cube_id) const;

    [[nodiscard]]
    bool contains_subsumed(const Cube& cube) const;

    [[nodiscard]]
    bool contains(const flat_state& vec) const;

    [[nodiscard]]
    vector<size_t>* data()
    {
        return &cubes_;
    }

    [[nodiscard]]
    const vector<size_t>* data() const
    {
        return &cubes_;
    }

private:
    vector<size_t> cubes_;
    CubeDatabase* db_;
    size_t frame_idx_;
};

class FramesInitializer {
public:
    FramesInitializer(CubeDatabase* cube_db, segmented_vector<Frame>* frames);

    void operator()();

private:
    CubeDatabase* cube_db_;
    segmented_vector<Frame>* frames_;
};

class FramesChecker {
public:
    FramesChecker(CubeDatabase* cube_db, segmented_vector<Frame>* frames);

    [[nodiscard]]
    bool operator()(const flat_state& state, size_t frame_idx) const;

private:
    // CubeDatabase* cube_db_;
    segmented_vector<Frame>* frames_;
};

class CubeInserter {
public:
    CubeInserter(CubeDatabase* cube_db, segmented_vector<Frame>* frames);

    size_t operator()(Cube cube, size_t frame_idx);

protected:
    CubeDatabase* cube_db_;
    segmented_vector<Frame>* frames_;
};

} // namespace police::ic3
