#pragma once

#include "police/base_types.hpp"
#include "police/storage/vector.hpp"
#include "police/verifiers/ic3/syntactic/frames_storage.hpp"

#include <limits>

namespace police::ic3::syntactic {

struct HyperArc {
    HyperArc(vector<size_t> successors, size_t label, size_t action_idx)
        : successors(std::move(successors))
        , label(label)
        , action_idx(action_idx)
    {
    }

    vector<size_t> successors;
    size_t label;
    size_t action_idx;

    size_t arc_id = -1;
    size_t post_id = -1;
};

class SyntacticAbstraction {
public:
    static constexpr size_t INF_FRAME = std::numeric_limits<size_t>::max() - 1;

    explicit SyntacticAbstraction(size_t num_vars);

    [[nodiscard]]
    size_t num_nodes() const
    {
        return node_arcs_.size();
    }

    [[nodiscard]]
    const vector<size_t>& get_in_arcs(size_t node) const
    {
        return as_successor_[node];
    }

    [[nodiscard]]
    vector<size_t>& get_node_arcs(size_t node)
    {
        return node_arcs_[node];
    }

    [[nodiscard]]
    const HyperArc& get_arc(size_t arc_id) const
    {
        return arcs_[arc_id];
    }

    void mark_arc_pruned(size_t arc_id);

    void notify_new_node();

    void add_arc(size_t src, HyperArc arc, Cube post);

    void add_successor(const HyperArc& arc, size_t successor);

    void apply_id_remap(const vector<size_t>& new_ids);

    void recompute_hmax();

    [[nodiscard]]
    size_t get_h_value(size_t node) const;

    void set_h_value(size_t node, size_t h_value);

    size_t recompute_h_value(size_t node);

    [[nodiscard]]
    const vector<size_t>& get_h_values() const
    {
        return hmax_;
    }

    template <typename Callback>
    void forall_arcs_where_necessarily(
        const Cube& necessary_condition,
        Callback callback)
    {
        post_conditions_.forall_that_subsume(
            necessary_condition,
            [&](size_t post_id) {
                for (size_t arc_id : post_to_arcs_[post_id]) {
                    callback(arcs_[arc_id]);
                }
            });
    }

    [[nodiscard]]
    const Cube& get_post_condition(const HyperArc& arc);

private:
    vector<HyperArc> arcs_;
    vector<vector<size_t>> node_arcs_;
    vector<size_t> hmax_;

    FramesStorage post_conditions_;
    vector<vector<size_t>> post_to_arcs_;
    vector<vector<size_t>> as_successor_;
    vector<size_t> num_successors_;
    vector<size_t> source_;

    bool clean_post_conditions_ = false;
};

} // namespace police::ic3::syntactic
