#include "police/verifiers/ic3/syntactic/frame_adder.hpp"

#include "police/macros.hpp"
#include "police/verifiers/ic3/syntactic/frames_storage.hpp"
#include "police/verifiers/ic3/start_generator.hpp"

namespace police::ic3::syntactic {

FrameAdder::FrameAdder(
    FramesStorage* frames,
    SyntacticAbstraction* abstraction,
    StartGenerator start_generator,
    size_t* cur_frame)
    : frames_(frames)
    , abstraction_(abstraction)
    , start_generator_(std::move(start_generator))
    , cur_frame_(cur_frame)
{
}

bool FrameAdder::operator()()
{
    const vector<size_t> h_old = abstraction_->get_h_values();

    // add new frame and clear start state generator
    ++*cur_frame_;
    start_generator_.clear();

    // recompute frame indices according to abstract transitions
    abstraction_->recompute_hmax();

#if POLICE_VERBOSITY > 2
    for (auto i = 0u; i < frames_->num_cubes(); ++i) {
        const size_t f_cur = h_old[i];
        const size_t f_new = abstraction_->get_h_value(i);
        POLICE_DEBUG_IFMSG(
            f_cur != f_new,
            "pushing cube " << i << " from frame " << f_cur << " to " << f_new
                            << "\n");
    }
#endif

    // check convergence: if top layer size remains unchanged
    size_t not_converged = 0;

#if POLICE_VERBOSITY > 2
    size_t num_shifted = 0;
    size_t num_deleted = 0;
    size_t top_cubes = 0;
#endif

    // update cube's frame references (but cap values at the frame limit);
    // mark subsumed nodes in the process
    for (size_t i = 0; i < frames_->num_cubes(); ++i) {
        const size_t f_cur = h_old[i];
        const size_t f_new = abstraction_->get_h_value(i);
        assert(f_cur <= f_new);
        not_converged += (f_cur == (*cur_frame_ - 1)) * (f_new == f_cur);
#if POLICE_VERBOSITY > 2
        top_cubes += f_cur == *cur_frame_ - 1;
#endif
        if (f_cur != f_new || i >= old_num_cubes_) {
#if POLICE_VERBOSITY > 2
            num_shifted += f_cur != f_new;
#endif
            FramesStorage::Range rng(0, f_new);
            frames_->forall_that_subsume(frames_->at(i), rng, [&](size_t j) {
                if (i != j && !frames_->is_orphaned(j)) {
#if POLICE_VERBOSITY > 2
                    ++num_deleted;
#endif
                    frames_->remove(j);
                }
            });
        }
        if (f_new >= *cur_frame_) {
            start_generator_.set_blocked(frames_->at(i));
        }
    }

    POLICE_DEBUG_MSG(
        "top cube(s) " << not_converged << " / " << top_cubes
                       << " have not yet converged\n");

    if (not_converged == 0u) {
        return true;
    }

    POLICE_DEBUG_MSG(
        "shifted " << num_shifted << " cube(s) to a higher frame\n"
                   << std::flush);
    POLICE_DEBUG_MSG(
        "marked " << num_deleted << " subsumed cube(s) for deletion\n");

    // remove dangling cubes (cubes that got subsumed by newly learned ones)
    if (frames_->is_fragmented()) {
        POLICE_DEBUG_MSG(
            "removing " << frames_->num_orphans() << " dangling cubes\n");
        auto new_ids = frames_->defragment();
        abstraction_->apply_id_remap(new_ids);
        POLICE_DEBUG_MSG("remaining cubes " << frames_->num_cubes() << "\n");
    }

    old_num_cubes_ = frames_->num_cubes();

    return false;
}

} // namespace police::ic3::syntactic
