#include "police/arguments.hpp"
#include "police/global_arguments.hpp"
#include "police/macros.hpp"
#include "police/model.hpp"
#include "police/option.hpp"
#include "police/utils/io.hpp"
#include "police/variable_order_chooser.hpp"
#include "police/verifiers/ic3/syntactic/policy_reasoner_nnlp.hpp"
#include "police/verifiers/ic3/syntactic/policy_reasoner_veritas.hpp"

#include <memory>

namespace police::ic3::syntactic {

namespace {
PointerOption<PolicyReasoner> _opt(
    "greedy_minimize",
    [](const Arguments& args) -> std::shared_ptr<PolicyReasoner> {
        const Model& model = args.get_model();
        const auto get_order = args.get_ptr<VariableOrderChooser>("order");
        const auto order = get_order->get_variable_order();
        std::cout << "Policy reasoner's greedy variable ordering: "
                  << print_sequence(order) << std::endl;
        if (args.has_nn_policy()) {
            const auto& policy = args.get_nn_policy();
            if (args.applicability_masking) {
#if 1
                POLICE_INTERNAL_ERROR(
                    "NNLP policy reasoner does currently not "
                    "support applicability masking");
#else
                return std::make_shared<PolicyReasonerNNLPMasked>(
                    *args.get_ptr<NNLPFactory>("lp"),
                    &model,
                    &policy,
                    order,
                    args.get<double>("max_total_time"),
                    args.get<double>("max_step_time"));
#endif
            }
            return std::make_shared<PolicyReasonerNNLP>(
                *args.get_ptr<NNLPFactory>("lp"),
                &model,
                &policy,
                order,
                args.get<double>("max_total_time"),
                args.get<double>("max_step_time"));
        } else {
            assert(args.has_addtree_policy());
            const auto& policy = args.get_addtree_policy();
            if (args.applicability_masking) {
                return std::make_shared<PolicyReasonerVeritasMasked>(
                    model,
                    policy,
                    order);
            }
            return std::make_shared<PolicyReasonerVeritas>(
                model,
                policy,
                order);
        }
    },
    [](const GlobalArguments& globals, ArgumentsDefinition& defs) {
        if (globals.has_nn_policy()) {
            defs.add_ptr_argument<NNLPFactory>("lp");
            defs.add_argument<double>(
                "max_total_time",
                "",
                std::to_string(std::numeric_limits<int>::max()));
            defs.add_argument<double>(
                "max_step_time",
                "",
                std::to_string(100'000'000.));
        }
        defs.add_ptr_argument<VariableOrderChooser>(
            "order",
            "Order in which to remove variables",
            "default");
    });
} // namespace

} // namespace police::ic3::syntactic
