#pragma once

#include "police/action.hpp"
#include "police/model.hpp"
#include "police/successor_generator/applicable_actions_generator.hpp"
#include "police/successor_generator/determinize_outcomes_adapator.hpp"
#include "police/successor_generator/successor_generator.hpp"
#include "police/utils/io.hpp"
#include "police/verifiers/ic3/syntactic/policy_reasons.hpp"
#include <algorithm>
#include <ranges>

namespace police::ic3::syntactic {

template <typename ApplicableActionsGenerator, typename Policy>
class PrunedAopsGenerator {
public:
    PrunedAopsGenerator(
        PolicyReasons* reasons,
        const Model* model,
        ApplicableActionsGenerator aops_gen,
        Policy policy)
        : reasons_(reasons)
        , aops_gen_(std::move(aops_gen))
        , policy_(std::move(policy))
        , model_(model)
    {
    }

    [[nodiscard]]
    successor_generator::ApplicableActions operator()(const flat_state& state)
    {
        auto aops = generate_actions(state);
        setify_labels();
        selected_ = policy_(state, labels_);
        if (static_cast<int_t>(state[state.size() - 1]) == 0) {
            apply_policy_filter(aops);
        }
        return aops;
    }

    [[nodiscard]]
    bool is_selected(size_t label) const
    {
        return label == SILENT_ACTION || selected_ == label;
    }

    [[nodiscard]]
    size_t get_selected() const
    {
        return selected_;
    }

private:
    successor_generator::ApplicableActions
    generate_actions(const flat_state& state)
    {
        successor_generator::ApplicableActions aops = aops_gen_(state);
        size_t j = 0;
        labels_.clear();
        for (size_t i = 0; i < aops.size(); ++i) {
            assert(aops[i] < model_->actions.size());
            const auto& action = model_->actions[aops[i]];
            if (action.label == SILENT_ACTION) {
                aops[j] = aops[i];
                ++j;
            } else if (!reasons_->is_blocked(state, action.label)) {
                aops[j] = aops[i];
                labels_.push_back(action.label);
                ++j;
            }
        }
        aops.erase(aops.begin() + j, aops.end());
        return aops;
    }

    void setify_labels()
    {
        std::sort(labels_.begin(), labels_.end());
        labels_.erase(
            std::unique(labels_.begin(), labels_.end()),
            labels_.end());
    }

    void apply_policy_filter(successor_generator::ApplicableActions& aops)
    {
        size_t i = 0;
        for (size_t j = 0; j < aops.size(); ++j) {
            const auto& action = model_->actions[aops[j]];
            if (is_selected(action.label)) {
                aops[i] = aops[j];
                ++i;
            }
        }
        aops.erase(aops.begin() + i, aops.end());
    }

    vector<size_t> labels_;
    size_t selected_ = SILENT_ACTION;

    PolicyReasons* reasons_;
    ApplicableActionsGenerator aops_gen_;
    Policy policy_;

    const Model* model_;
};

template <typename ApplicableActionsGenerator, typename Policy>
class PruningSuccessorGenerator
    : private successor_generator::determinize_outcomes_adapator<
          successor_generator::SuccessorGenerator<
              PrunedAopsGenerator<ApplicableActionsGenerator, Policy>>> {
public:
    PruningSuccessorGenerator(
        const vector<Action>* actions,
        PrunedAopsGenerator<ApplicableActionsGenerator, Policy> aops_gen)
        : successor_generator::determinize_outcomes_adapator<
              successor_generator::SuccessorGenerator<
                  PrunedAopsGenerator<ApplicableActionsGenerator, Policy>>>(
              successor_generator::SuccessorGenerator<
                  PrunedAopsGenerator<ApplicableActionsGenerator, Policy>>(
                  actions,
                  std::move(aops_gen)))
    {
    }

    auto operator()(const flat_state& state)
    {
        auto successors = successor_generator::determinize_outcomes_adapator<
            successor_generator::SuccessorGenerator<
                PrunedAopsGenerator<ApplicableActionsGenerator, Policy>>>::
        operator()(state);
        const size_t bvar = state.size() - 1;
        const int budget = std::max(0, static_cast<int_t>(state[bvar]) - 1);
        const int non_zero = static_cast<int_t>(state[bvar]) != 0;
        const auto selected = this->aops_gen_.get_selected();
        return successors |
               std::ranges::views::transform(
                   [selected, non_zero, budget](
                       successor_generator::Successor succ) {
                       succ.state[succ.state.size() - 1] = Value(
                           budget + non_zero * (succ.label == SILENT_ACTION ||
                                                succ.label == selected));
                       return succ;
                   });
    }
};

} // namespace police::ic3::syntactic
