#pragma once

#include "police/verifiers/search/concepts.hpp"
#include "police/verifiers/search/label_id.hpp"
#include "police/verifiers/search/node_id.hpp"
#include "police/verifiers/search/node_info.hpp"
#include "police/verifiers/search/node_info_container.hpp"
#include "police/verifiers/search/path.hpp"
#include "police/verifiers/search/queues.hpp"

#include <algorithm>
#include <cassert>
#include <concepts>
#include <optional>
#include <type_traits>
#include <utility>

namespace police::search {

template <typename Q, typename StateType>
concept best_first_search_queue = requires(Q q) {
    { q.push(std::declval<StateType>()) } -> std::convertible_to<bool>;
    { q.pop() } -> std::convertible_to<search::NodeId>;
    { q.empty() } -> std::convertible_to<bool>;
    q.clear();
};

template <
    typename StateType,
    search::state_to_id_map<StateType> StateToId,
    search::id_to_state_map<StateType> IdToState,
    search::successor_generator<StateType> SuccessorsGenerator,
    search::state_predicate<StateType> GoalCondition,
    search::state_callable<StateType> OnNewState,
    best_first_search_queue<StateType> Queue>
struct best_first_search {
    static_assert(
        std::is_convertible_v<
            decltype(std::declval<StateToId>()(std::declval<StateType>())),
            search::NodeId>);

    using NodeInfo =
        search::BaseNodeInfo<search::NodePathInfo<search::PlainNodeInfo>>;

    constexpr best_first_search(
        StateToId state_to_id = StateToId(),
        IdToState id_to_state = IdToState(),
        SuccessorsGenerator succ_gen = SuccessorsGenerator(),
        GoalCondition goal = GoalCondition(),
        OnNewState on_new_state = OnNewState(),
        Queue queue = Queue()) noexcept
        : state_to_id_(std::move(state_to_id))
        , id_to_state_(std::move(id_to_state))
        , succ_gen_(std::move(succ_gen))
        , is_goal_(std::move(goal))
        , new_state_listener_(std::move(on_new_state))
        , queue_(std::move(queue))
    {
    }

    [[nodiscard]]
    constexpr std::optional<search::Path> operator()(const StateType& state)
    {
        push_initial_state(state);
        for (; !queue_.empty() && !step(););
        return goal_path_;
    }

    [[nodiscard]]
    constexpr search::NodeId current_node() const
    {
        return node_id_;
    }

    [[nodiscard]]
    constexpr search::LabelId current_label() const
    {
        return label_id_;
    }

    void reset()
    {
        queue_.clear();
        search::NodeInfoContainer<NodeInfo>().swap(node_infos_);
        goal_path_ = std::nullopt;
        node_id_ = search::UNDEFINED_NODE;
        label_id_ = search::UNDEFINED_LABEL;
    }

protected:
    constexpr bool push_initial_state(const StateType& state)
    {
        assert(queue_.empty());
        goal_path_ = std::nullopt;
        return push_into_queue(
            search::UNDEFINED_NODE,
            search::UNDEFINED_LABEL,
            state);
    }

    constexpr void set_search_result(search::NodeId goal_node)
    {
        goal_path_ = traceback(goal_node);
    }

    constexpr bool step()
    {
        assert(!queue_.empty());
        node_id_ = queue_.pop();
        NodeInfo& node_info = node_infos_[node_id_];
        assert(!node_info.is_new());
        if (node_info.is_closed()) {
            return false;
        }
        if (node_info.is_goal()) {
            set_search_result(node_id_);
            return true;
        }
        node_info.set_expansion_status(true);
        StateType state = id_to_state_(node_id_);
        auto successor_range = succ_gen_(state);
        auto it = successor_range.begin();
        const auto end = successor_range.end();
        for (; it != end; ++it) {
            const auto& succ = *it;
            label_id_ = succ.label;
            push_into_queue(node_id_, succ.label, succ.state);
        }
        return false;
    }

    constexpr bool push_into_queue(
        search::NodeId parent,
        search::LabelId label,
        const StateType& succ_state)
    {
        search::NodeId node_id = state_to_id_(succ_state);
        NodeInfo& node_info = node_infos_[node_id];
        if (node_info.is_new()) {
            new_state_listener_(succ_state);
            node_info.set_parent(parent, label);
            node_info.set_goal(is_goal_(succ_state));
            node_info.set_expansion_status(!queue_.push(succ_state));
            return false;
        }
        return true;
    }

    constexpr search::Path traceback(search::NodeId node_id) const
    {
        assert(node_id != search::UNDEFINED_NODE);
        search::Path path;
        path.emplace_back(node_id);
        const NodeInfo* node_info = &node_infos_[node_id];
        while (node_info->get_parent() != search::UNDEFINED_NODE) {
            path.emplace_back(node_info->get_parent(), node_info->get_label());
            node_info = &node_infos_[node_info->get_parent()];
        }
        std::reverse(path.begin(), path.end());
        return path;
    }

    std::optional<search::Path> goal_path_ = std::nullopt;
    search::NodeId node_id_ = search::UNDEFINED_NODE;
    search::LabelId label_id_ = search::UNDEFINED_LABEL;
    search::NodeInfoContainer<NodeInfo> node_infos_;

    StateToId state_to_id_;
    IdToState id_to_state_;
    SuccessorsGenerator succ_gen_;
    GoalCondition is_goal_;
    OnNewState new_state_listener_;
    Queue queue_;
};

template <
    typename StateType,
    search::state_to_id_map<StateType> StateToId,
    search::id_to_state_map<StateType> IdToState,
    search::successor_generator<StateType> SuccessorsGenerator,
    search::state_predicate<StateType> GoalCondition,
    search::state_callable<StateType> OnNewState,
    best_first_search_queue<StateType> Queue>
best_first_search<
    StateType,
    StateToId,
    IdToState,
    SuccessorsGenerator,
    GoalCondition,
    OnNewState,
    Queue>
create_best_first_search(
    StateToId state_to_id = StateToId(),
    IdToState id_to_state = IdToState(),
    SuccessorsGenerator succ_gen = SuccessorsGenerator(),
    GoalCondition goal = GoalCondition(),
    OnNewState on_new_state = OnNewState(),
    Queue queue = Queue())
{
    return {
        std::move(state_to_id),
        std::move(id_to_state),
        std::move(succ_gen),
        std::move(goal),
        std::move(on_new_state),
        std::move(queue)};
}

template <
    typename StateType,
    search::state_to_id_map<StateType> StateToId,
    search::id_to_state_map<StateType> IdToState,
    search::successor_generator<StateType> SuccessorsGenerator,
    search::state_predicate<StateType> GoalCondition,
    search::state_callable<StateType> OnNewState>
best_first_search<
    StateType,
    StateToId,
    IdToState,
    SuccessorsGenerator,
    GoalCondition,
    OnNewState,
    search::FifoQueue<StateType, StateToId>>
create_breadth_first_search(
    StateToId state_to_id = StateToId(),
    IdToState id_to_state = IdToState(),
    SuccessorsGenerator succ_gen = SuccessorsGenerator(),
    GoalCondition goal = GoalCondition(),
    OnNewState on_new_state = OnNewState())
{
    return {
        std::move(state_to_id),
        std::move(id_to_state),
        std::move(succ_gen),
        std::move(goal),
        std::move(on_new_state),
        search::FifoQueue<StateType, StateToId>{std::move(state_to_id)}};
}

} // namespace police::search
