#pragma once

#include "police/verifiers/search/label_id.hpp"
#include "police/verifiers/search/node_id.hpp"
#include "police/utils/bit_operations.hpp"
#include <cstdint>

namespace police::search {

using status_flags_t = std::uint8_t;

struct PlainNodeInfo {
    constexpr static status_flags_t STATUS_BITS = 0;
};

template <typename BaseInfo>
struct BaseNodeInfo final : public BaseInfo {
    static_assert(BaseInfo::STATUS_BITS + 3 <= sizeof(status_flags_t) * 8);
    constexpr static status_flags_t EXPANSION_STATUS_MASK =
        bits::pattern<std::uint8_t>(2, BaseInfo::STATUS_BITS);
    constexpr static status_flags_t ZEROED = 0u;
    constexpr static status_flags_t OPEN(status_flags_t b = 1)
    {
        return b << (BaseInfo::STATUS_BITS);
    }
    constexpr static status_flags_t CLOSED(status_flags_t b = 1)
    {
        return b << (BaseInfo::STATUS_BITS + 1);
    }
    constexpr static status_flags_t GOAL(status_flags_t b = 1)
    {
        return b << (BaseInfo::STATUS_BITS + 2);
    }

    [[nodiscard]]
    constexpr bool is_new() const
    {
        return status == ZEROED;
    }

    [[nodiscard]]
    constexpr bool is_open() const
    {
        return status & OPEN();
    }

    [[nodiscard]]
    constexpr bool is_closed() const
    {
        return status & CLOSED();
    }

    [[nodiscard]]
    constexpr bool is_goal() const
    {
        return status & GOAL();
    }

    constexpr void set_expansion_status(bool closed)
    {
#ifndef NDEBUG
        const bool goal = is_goal();
#endif
        status =
            (status & ~EXPANSION_STATUS_MASK) | OPEN(!closed) | CLOSED(closed);
        assert(goal == is_goal());
        assert(is_open() == !closed);
        assert(is_closed() == closed);
        assert(!is_new());
    }

    constexpr void set_goal(bool is_goal)
    {
#ifndef NDEBUG
        const bool closed = is_closed();
        const bool open = is_open();
#endif
        status = status | GOAL(is_goal);
        assert(is_closed() == closed);
        assert(is_open() == open);
        assert(this->is_goal() == is_goal);
    }

    status_flags_t status = ZEROED;
};

template <typename BaseInfo>
struct NodePathInfo : public BaseInfo {
    using BaseInfo::STATUS_BITS;

    constexpr void set_parent(NodeId parent, LabelId label)
    {
        this->parent = parent;
        this->label = label;
    }

    constexpr NodeId get_parent() const { return parent; }

    constexpr LabelId get_label() const { return label; }

    NodeId parent = UNDEFINED_NODE;
    LabelId label = UNDEFINED_LABEL;
};

} // namespace police::search
