#include "police/expressions/constants.hpp"
#include "police/expressions/expression.hpp"
#include "police/jani/parser/expression.hpp"
#include "police/jani/parser/types.hpp"

#include <catch2/catch.hpp>
#include <memory>
#include <nlohmann/json.hpp>
#include <ranges>
#include <string_view>
#include <type_traits>

namespace police::test {

nlohmann::json
make_unary_operation(const std::string& op, nlohmann::json expr = "a");

nlohmann::json make_operation(
    const std::string& op,
    nlohmann::json left = "a",
    nlohmann::json right = "b");

template <typename T>
nlohmann::json to_list(const T& t)
{
    nlohmann::json::array_t l;
    auto tojs =
        t | std::ranges::views::transform([](auto&& x) -> nlohmann::json {
            using S = std::decay_t<decltype(x)>;
            if constexpr (std::is_convertible_v<S, std::string_view>) {
                if (std::string_view(x) == "null") {
                    return "null"_json;
                }
            }
            return nlohmann::json(x);
        });
    l.insert(l.end(), tojs.begin(), tojs.end());
    return l;
}

template <typename Expected, typename T>
std::shared_ptr<Expected> cast(const std::shared_ptr<T>& ptr)
{
    auto x = std::dynamic_pointer_cast<Expected>(ptr);
    REQUIRE(x != nullptr);
    return x;
}

template <typename T>
T type_cast(const police::jani::parser::Type& t)
{
    auto cast = [](auto&& t) -> std::optional<T> {
        using S = std::decay_t<decltype(t)>;
        if constexpr (std::is_same_v<S, T>) {
            return {t};
        }
        return std::nullopt;
    };
    auto cast_t = std::visit(cast, t);
    REQUIRE(cast_t.has_value());
    return cast_t.value();
}

template <typename ExpectedType>
std::shared_ptr<ExpectedType> test_expr_parser(const nlohmann::json& js)
{
    auto parsed = police::jani::parser::expression_schema()(js, false);
    REQUIRE(parsed.has_value());
    return cast<ExpectedType>(parsed.value().base());
}

template <typename T>
void require_is_constant(const police::expressions::Expression& expr, T value)
{
    REQUIRE(expr.is_same(expressions::MakeConstant()(value)));
}

template <typename T>
void test_constant(T value)
{
    nlohmann::json js = value;
    auto e = police::jani::parser::expression_schema()(js, true);
    REQUIRE(e.has_value());
    require_is_constant(e.value(), value);
}

void require_is_identifier(
    const police::expressions::Expression& expr,
    std::string_view identifier);

} // namespace police::test
