#include "helpers.hpp"
#include "police/expressions/constants.hpp"
#include "police/jani/parser/automaton.hpp"

#include <catch2/catch.hpp>
#include <nlohmann/json.hpp>

namespace {
using json = nlohmann::json;
namespace parser = police::jani::parser;

parser::Automaton test_automaton(const json& js)
{
    auto x = police::jani::parser::automaton_schema()(js, true);
    REQUIRE(x.has_value());
    return x.value();
}

json automaton(json edges)
{
    auto obj =
        R"({"name": "auto", "locations": [{"name": "loc0"}, {"name": "loc1"}], "initial-locations": ["loc0"]})"_json;
    obj["edges"] = edges;
    return obj;
}

} // namespace

TEST_CASE("Parse automaton: single location", "[jani][parser][automaton]")
{
    auto a = test_automaton(
        R"({"name": "auto", "locations": [{"name": "loc0"}], "initial-locations": ["loc0"], "edges": []})"_json);
    REQUIRE(a.name == "auto");
    REQUIRE(a.locations.size() == 1);
    REQUIRE(a.locations[0].name == "loc0");
    REQUIRE(a.initial_locations.size() == 1);
    REQUIRE(a.initial_locations[0] == "loc0");
    REQUIRE(a.edges.empty());
}

TEST_CASE("Parse automaton: simple edge", "[jani][parser][automaton]")
{
    auto a = test_automaton(automaton(
        R"([{"location": "loc0", "destinations": [{"location": "loc1"}]}])"_json));
    REQUIRE(a.name == "auto");
    REQUIRE(a.locations.size() == 2);
    REQUIRE(a.initial_locations.size() == 1);
    REQUIRE(a.edges.size() == 1);
    REQUIRE(a.edges[0].location == "loc0");
    REQUIRE(!a.edges[0].guard.has_value());
    REQUIRE(a.edges[0].destinations.size() == 1);
    REQUIRE(a.edges[0].destinations[0].location == "loc1");
    REQUIRE(a.edges[0].destinations[0].assignments.empty());
}

TEST_CASE("Parse automaton: edge with guard", "[jani][parser][automaton]")
{
    auto a = test_automaton(automaton(
        R"([{"location": "loc0", "guard": {"exp": true}, "destinations": [{"location": "loc1"}]}])"_json));
    REQUIRE(a.edges.size() == 1);
    REQUIRE(a.edges[0].location == "loc0");
    REQUIRE(a.edges[0].guard.has_value());
    REQUIRE(a.edges[0].guard.value().is_same(
        police::expressions::MakeConstant()(true)));
    REQUIRE(a.edges[0].destinations.size() == 1);
    REQUIRE(a.edges[0].destinations[0].location == "loc1");
    REQUIRE(a.edges[0].destinations[0].assignments.empty());
}

TEST_CASE(
    "Parse automaton: edge with multiple destinations",
    "[jani][parser][automaton]")
{
    auto a = test_automaton(automaton(
        R"([{"location": "loc0", "guard": {"exp": true}, "destinations": [{"location": "loc1"}, {"location": "loc1"}]}])"_json));
    REQUIRE(a.edges.size() == 1);
    REQUIRE(a.edges[0].location == "loc0");
    REQUIRE(a.edges[0].guard.has_value());
    REQUIRE(a.edges[0].guard.value().is_same(
        police::expressions::MakeConstant()(true)));
    REQUIRE(a.edges[0].destinations.size() == 2);
    REQUIRE(a.edges[0].destinations[0].location == "loc1");
    REQUIRE(a.edges[0].destinations[0].assignments.empty());
    REQUIRE(a.edges[0].destinations[1].location == "loc1");
    REQUIRE(a.edges[0].destinations[1].assignments.empty());
}

TEST_CASE("Parse automaton: edge with assignment", "[jani][parser][automaton]")
{
    auto a = test_automaton(automaton(
        R"([{"location": "loc0", "guard": {"exp": true}, "destinations": [{"location": "loc1", "assignments": [{"ref": "x", "value": true}]}, {"location": "loc1"}]}])"_json));
    REQUIRE(a.edges.size() == 1);
    REQUIRE(a.edges[0].location == "loc0");
    REQUIRE(a.edges[0].guard.has_value());
    REQUIRE(a.edges[0].guard.value().is_same(
        police::expressions::MakeConstant()(true)));
    REQUIRE(a.edges[0].destinations.size() == 2);
    REQUIRE(a.edges[0].destinations[0].location == "loc1");
    REQUIRE(a.edges[0].destinations[0].assignments.size() == 1);
    REQUIRE(a.edges[0].destinations[0].assignments[0].ref == "x");
    police::test::require_is_constant(
        a.edges[0].destinations[0].assignments[0].value,
        true);
}

TEST_CASE(
    "Parse automaton: edge with multiple assignments",
    "[jani][parser][automaton]")
{
    auto a = test_automaton(automaton(
        R"([{"location": "loc0", "guard": {"exp": true}, "destinations": [{"location": "loc1", "assignments": [{"ref": "x", "value": true}, {"ref": "y", "value": 123}]}]}])"_json));
    REQUIRE(a.edges.size() == 1);
    REQUIRE(a.edges[0].location == "loc0");
    REQUIRE(a.edges[0].guard.has_value());
    REQUIRE(a.edges[0].guard.value().is_same(
        police::expressions::MakeConstant()(true)));
    REQUIRE(a.edges[0].destinations.size() == 1);
    REQUIRE(a.edges[0].destinations[0].location == "loc1");
    REQUIRE(a.edges[0].destinations[0].assignments.size() == 2);
    REQUIRE(a.edges[0].destinations[0].assignments[0].ref == "x");
    police::test::require_is_constant(
        a.edges[0].destinations[0].assignments[0].value,
        true);
    REQUIRE(a.edges[0].destinations[0].assignments[1].ref == "y");
    police::test::require_is_constant(
        a.edges[0].destinations[0].assignments[1].value,
        123);
}
