#include "helpers.hpp"
#include "police/expressions/expression.hpp"
#include "police/expressions/filter_property.hpp"
#include "police/expressions/property.hpp"
#include "police/expressions/state_predicate_property.hpp"
#include "police/expressions/state_property.hpp"
#include "police/jani/parser/language.hpp"
#include "police/jani/parser/property.hpp"

#include <catch2/catch.hpp>
#include <memory>
#include <nlohmann/json.hpp>
#include <optional>
#include <string_view>

namespace {
using json = nlohmann::json;
using namespace police::test;

const police::expressions::Expression&
state_expression_cast(police::expressions::PropertyPtr prop)
{
    return cast<police::expressions::StateProperty>(prop)->expr;
}

json prop(std::string_view name, const json& expr)
{
    json obj;
    obj[police::jani::parser::lang::NAME] = name;
    obj[police::jani::parser::lang::EXPRESSION] = expr;
    return obj;
}

json filter_prop(
    std::string_view name,
    std::string_view fun,
    const json& values,
    const json& states)
{
    json obj;
    obj[police::jani::parser::lang::OP] = police::jani::parser::lang::FILTER;
    obj[police::jani::parser::lang::FUN] = fun;
    obj[police::jani::parser::lang::VALUES] = values;
    obj[police::jani::parser::lang::STATES] = states;
    return prop(name, obj);
}

json state_pred(std::string_view pred)
{
    json obj;
    obj[police::jani::parser::lang::OP] = pred;
    return obj;
}

template <typename ExpProperty>
std::shared_ptr<ExpProperty>
test_property(const json& js, std::string_view exp_name)
{
    auto raw_prop = police::jani::parser::property_schema()(js, false);
    REQUIRE(raw_prop.has_value());
    REQUIRE(raw_prop.value().name == exp_name);
    auto prop = std::dynamic_pointer_cast<ExpProperty>(raw_prop.value().expr);
    REQUIRE(prop != nullptr);
    return prop;
}

} // namespace

TEST_CASE("Parse filter property", "[jani][parser][property]")
{
    auto prop = test_property<police::expressions::FilterProperty>(
        filter_prop("p", police::jani::parser::lang::FORALL, "x", "x"),
        "p");
    REQUIRE(prop->fun == police::expressions::FilterProperty::Function::FORALL);
    require_is_identifier(state_expression_cast(prop->states), "x");
    require_is_identifier(state_expression_cast(prop->values), "x");
}

TEST_CASE("Parse filter initial state property", "[jani][parser][property]")
{
    auto prop = test_property<police::expressions::FilterProperty>(
        filter_prop(
            "ppp",
            police::jani::parser::lang::EXISTS,
            "x",
            state_pred(police::jani::parser::lang::INITIAL)),
        "ppp");
    REQUIRE(prop->fun == police::expressions::FilterProperty::Function::EXISTS);
    require_is_identifier(state_expression_cast(prop->values), "x");
    auto pred = cast<police::expressions::StatePredicateProperty>(prop->states);
    REQUIRE(
        pred->predicate ==
        police::expressions::StatePredicateProperty::Predicate::INITIAL);
}
