#include "police/base_types.hpp"
#include "police/layer_bounds.hpp"
#include "police/linear_perturbation_analyzer.hpp"
#include "police/storage/ffnn.hpp"

#include <catch2/catch.hpp>
#include <iostream>

namespace {
using namespace police;
using police::operator<<;

LayerBounds<real_t> analyze(
    const FeedForwardNeuralNetwork<>& net,
    const LayerBounds<real_t>& bounds)
{
    LinearPerturbationAnalyzer<real_t> analyzer(net);
    auto result = analyzer(bounds);
    std::cout << "in: " << bounds << "\n";
    for (auto l = 0u; l < analyzer.layers().size(); ++l) {
        std::cout << "L" << l << ": " << analyzer.layers()[l].get_bounds()
                  << "\n";
    }
    std::cout << std::flush;
    return result;
}

FeedForwardNeuralNetwork<> network_1to1_2l()
{
    FeedForwardNeuralNetwork<> net;
    net.layers.resize(2);

    auto& l0 = net.layers[0];
    l0.biases.resize(1, 0);
    l0.weights.resize(1, {1.});

    auto& l1 = net.layers[1];
    l1.biases.resize(1, 1);
    l1.weights.resize(1, {1.});

    return net;
}

FeedForwardNeuralNetwork<> network_2to1_2l()
{
    FeedForwardNeuralNetwork<> net;
    net.layers.resize(2);

    auto& l0 = net.layers[0];
    l0.biases.resize(2, 0);
    l0.weights.resize(2);
    l0.weights[0] = {1., 2.};
    l0.weights[1] = {2., 1.};

    auto& l1 = net.layers[1];
    l1.biases.resize(1, 0);
    l1.weights.resize(1, {1., -1.});

    return net;
}

FeedForwardNeuralNetwork<> network_3l_neurips()
{
    FeedForwardNeuralNetwork<> net;
    net.layers.resize(3);

    auto& l0 = net.layers[0];
    l0.biases.resize(2, 0);
    l0.weights.resize(2);
    l0.weights[0] = {2., 1.};
    l0.weights[1] = {-3., 4.};

    auto& l1 = net.layers[1];
    l1.biases.resize(2, 0);
    l1.weights.resize(2);
    l1.weights[0] = {4, -2.};
    l1.weights[1] = {2., 1.};

    auto& l2 = net.layers[2];
    l2.biases.resize(1, 0);
    l2.weights.resize(1);
    l2.weights[0] = {-2., 1.};

    return net;
}

} // namespace

TEST_CASE("Test LIPA 2L 1x1 inactive", "[lipa]")
{
    auto net = network_1to1_2l();
    LayerBounds<real_t> in(1);
    in.set_bounds(0, 0, 1);
    auto out = analyze(net, in);
    std::cout << out.lb(0) << " <= o <= " << out.ub(0) << std::endl;
    REQUIRE(out.lb(0) >= 0);
}

TEST_CASE("Test LIPA 2L 1x1 active", "[lipa]")
{
    auto net = network_1to1_2l();
    LayerBounds<real_t> in(1);
    in.set_bounds(0, -1, 0);
    auto out = analyze(net, in);
    std::cout << out.lb(0) << " <= o <= " << out.ub(0) << std::endl;
    REQUIRE(out.lb(0) >= 0);
}

TEST_CASE("Test LIPA 2L 1x1 unstable", "[lipa]")
{
    auto net = network_1to1_2l();
    LayerBounds<real_t> in(1);
    in.set_bounds(0, -1, 1);
    auto out = analyze(net, in);
    std::cout << out.lb(0) << " <= o <= " << out.ub(0) << std::endl;
    REQUIRE(out.lb(0) >= 0);
}

TEST_CASE("Test LIPA 2L 2x1", "[lipa]")
{
    auto net = network_2to1_2l();
    LayerBounds<real_t> in(2);
    in.set_bounds(0, -1, 1);
    in.set_bounds(1, -1, 1);
    auto out = analyze(net, in);
    std::cout << out.lb(0) << " <= o <= " << out.ub(0) << std::endl;
    REQUIRE(out.lb(0) >= -1.75);
}

TEST_CASE("Test LIPA 3L (NeurIPS example)", "[lipa]")
{
    auto net = network_3l_neurips();
    LayerBounds<real_t> in(2);
    in.set_bounds(0, -2, 2);
    in.set_bounds(1, -1, 3);
    auto out = analyze(net, in);
    std::cout << out.lb(0) << " <= o <= " << out.ub(0) << std::endl;
    REQUIRE(out.ub(0) >= 24.);
    REQUIRE(out.lb(0) <= -36.);
}
