#if POLICE_GUROBI

#include "../numbers.hpp"
#include "police/lp.hpp"
#include "police/lp_gurobi.hpp"

#include <catch2/catch.hpp>

namespace {
using namespace police;
}

TEST_CASE(
    "LP with two indicator constraints is solvable",
    "[lp_interface][lp][gurobi]")
{
    GurobiLP lp(LPOptimizationKind::MAXIMIZE);

    const auto x =
        lp.add_variable(LPVariable(0, 100, 0, LPVariable::Type::REAL));
    const auto y =
        lp.add_variable(LPVariable(0, 100, 0, LPVariable::Type::REAL));
    const auto z = lp.add_variable(LPVariable(0, 1, 1, LPVariable::Type::BOOL));

    {
        LP::linear_constraint_type c(LP::linear_constraint_type::GREATER_EQUAL);
        c.insert(x, 1.);
        c.insert(y, -1.);
        lp.add_constraint(c);
    }

    {
        LP::linear_constraint_type c(LP::linear_constraint_type::LESS_EQUAL);
        c.insert(x, 1.);
        c.rhs = 25.;
        lp.add_constraint(LP::indicator_constraint_type(z, true, c));
    }

    {
        LP::linear_constraint_type c(LP::linear_constraint_type::GREATER_EQUAL);
        c.insert(y, 1.);
        c.rhs = 25;
        lp.add_constraint(LP::indicator_constraint_type(z, true, c));
    }

    const auto status = lp.solve();

    REQUIRE(status == LPStatus::OPTIMAL);

    auto model = lp.get_model();
    CHECK_THAT(
        static_cast<real_t>(model[x]),
        Catch::Matchers::WithinAbs(25, PRECISION));
    CHECK_THAT(
        static_cast<real_t>(model[y]),
        Catch::Matchers::WithinAbs(25, PRECISION));
    CHECK_THAT(
        static_cast<real_t>(model[z]),
        Catch::Matchers::WithinAbs(1, PRECISION));
}

TEST_CASE(
    "LP with two indicator constraints is solvable with indicator = 0",
    "[lp_interface][lp][gurobi]")
{
    GurobiLP lp(LPOptimizationKind::MAXIMIZE);

    const auto x =
        lp.add_variable(LPVariable(0, 100, 0, LPVariable::Type::REAL));
    const auto y =
        lp.add_variable(LPVariable(0, 100, 0, LPVariable::Type::REAL));
    const auto z = lp.add_variable(LPVariable(0, 1, 1, LPVariable::Type::BOOL));

    {
        LP::linear_constraint_type c(LP::linear_constraint_type::GREATER_EQUAL);
        c.insert(x, 1.);
        c.insert(y, -1.);
        lp.add_constraint(c);
    }

    {
        LP::linear_constraint_type c(LP::linear_constraint_type::LESS_EQUAL);
        c.insert(x, 1.);
        c.rhs = 20.;
        lp.add_constraint(LP::indicator_constraint_type(z, true, c));
    }

    {
        LP::linear_constraint_type c(LP::linear_constraint_type::GREATER_EQUAL);
        c.insert(y, 1.);
        c.rhs = 25;
        lp.add_constraint(LP::indicator_constraint_type(z, true, c));
    }

    const auto status = lp.solve();

    REQUIRE(status == LPStatus::OPTIMAL);

    auto model = lp.get_model();
    CHECK_THAT(
        static_cast<real_t>(model[z]),
        Catch::Matchers::WithinAbs(0, PRECISION));
}

TEST_CASE(
    "LP with two indicator constraints is infeasible",
    "[lp_interface][lp][gurobi]")
{
    GurobiLP lp(LPOptimizationKind::MAXIMIZE);

    const auto x =
        lp.add_variable(LPVariable(0, 100, 0, LPVariable::Type::REAL));
    const auto y =
        lp.add_variable(LPVariable(0, 100, 0, LPVariable::Type::REAL));
    const auto z = lp.add_variable(LPVariable(0, 1, 1, LPVariable::Type::BOOL));

    {
        LP::linear_constraint_type c(LP::linear_constraint_type::GREATER_EQUAL);
        c.insert(x, 1.);
        c.insert(y, -1.);
        lp.add_constraint(c);
    }

    {
        LP::linear_constraint_type c(LP::linear_constraint_type::LESS_EQUAL);
        c.insert(x, 1.);
        c.rhs = 20.;
        lp.add_constraint(LP::indicator_constraint_type(z, true, c));
    }

    {
        LP::linear_constraint_type c(LP::linear_constraint_type::GREATER_EQUAL);
        c.insert(y, 1.);
        c.rhs = 25;
        lp.add_constraint(LP::indicator_constraint_type(z, true, c));
    }

    {
        LP::linear_constraint_type c(LP::linear_constraint_type::GREATER_EQUAL);
        c.insert(z, 1.);
        c.rhs = 1.;
        lp.add_constraint(c);
    }

    const auto status = lp.solve();

    REQUIRE(status == LPStatus::INFEASIBLE);
}

#endif
