#include "police/storage/explicit_state_container.hpp"
#include "police/storage/flat_state.hpp"

#include <catch2/catch.hpp>

namespace {
using StateRegistry = police::explicit_state_container<>;

struct Bounds {
    int lower_bound;
    int upper_bound;
};

std::vector<Bounds> get_variables()
{
    std::vector<Bounds> bounds;
    bounds.emplace_back(-100, 100);
    bounds.emplace_back(0, 100);
    bounds.emplace_back(0, 1);
    bounds.emplace_back(0, 1);
    return bounds;
}

StateRegistry get_state_registry()
{
    const auto vars = get_variables();
    return StateRegistry(1, vars.begin(), vars.end());
}

police::flat_state make_state(double a, int b, int c, int d, int e)
{
    police::flat_state state(5);
    state[0] = police::Value(a);
    state[1] = police::Value(b);
    state[2] = police::Value(c);
    state[3] = police::Value(d);
    state[4] = police::Value(e);
    return state;
}

} // namespace

TEST_CASE("Empty state registry", "[storage][state_registry]")
{
    StateRegistry reg = get_state_registry();
    REQUIRE(reg.size() == 0u);
}

TEST_CASE("State registry manipulation", "[storage][state_registry]")
{
    StateRegistry reg = get_state_registry();
    const auto state1 = make_state(0.0, 0, 0, 0, 0);
    const auto state2 = make_state(32.125, 100, 50, 0, 0);
    const auto state3 = make_state(32.125, 100, 50, 1, 0);
    auto x = reg.insert(state1);

    SECTION("Insert state")
    {
        REQUIRE(x.second);
        REQUIRE(x.first == 0u);
        REQUIRE(reg.size() == 1);
    }

    SECTION("Lookup")
    {
        auto state = reg[0];
        REQUIRE(state == state1);
    }

    SECTION("Insert same state twice")
    {
        auto y = reg.insert(state1);
        REQUIRE(!y.second);
        REQUIRE(y.first == 0u);
        REQUIRE(reg.size() == 1);
    }

    SECTION("Insert another state")
    {
        auto y = reg.insert(state2);
        REQUIRE(y.second);
        REQUIRE(y.first == 1u);
        REQUIRE(reg.size() == 2);
        REQUIRE(reg[x.first] == state1);
        REQUIRE(reg[y.first] == state2);
    }

    SECTION("Insert two states")
    {
        auto y = reg.insert(state2);
        auto z = reg.insert(state3);
        REQUIRE(y.second);
        REQUIRE(z.second);
        REQUIRE(reg.size() == 3);
        auto yy = reg.insert(state2);
        REQUIRE((!yy.second && yy.first == y.first));
        auto zz = reg.insert(state3);
        REQUIRE((!zz.second && zz.first == z.first));
        REQUIRE(reg[z.first] == state3);
        REQUIRE(reg[y.first] == state2);
        REQUIRE(reg[x.first] == state1);
    }
}
