import torch
import torch.backends.cudnn as cudnn
import os
import argparse
import time
from utils import AverageMeter, cosine_annealing, logger, save_checkpoint, evaluation, setup_seed, classwise_evaluation
from torch.nn import functional as F
from cifar import CIFAR10Index
from torchvision.models import resnet18, resnet50
from torch import nn
from torchvision import transforms
from torch.utils.data import DataLoader
import kornia.augmentation as K
import numpy as np


def get_args():
    parser = argparse.ArgumentParser(description='EVAL')
    parser.add_argument('--backbone', type=str, default='resnet18', choices=['resnet18', 'resnet50', 'vgg19',
                                                                             'vit-b', 'densenet121', 'linear',
                                                                             '2nn', '3nn', 'lenet5', 'vit', 'wrn34-10'],
                        help='the model arch used in experiment')

    parser.add_argument('--dataset', default='cifar10', choices=['cifar10', 'cifar100', 'tinyimagenet',
                                                                 'miniimagenet', 'imagenet100'],
                        help='the dataset used in experiment')
    parser.add_argument('--data', type=str, default='data/CIFAR10', help='the directory of dataset')
    parser.add_argument('--num-classes', default=10, type=int, help='the number of classes in the dataset')
    parser.add_argument('--batch-size', type=int, default=128)
    parser.add_argument('--num-workers', type=int, default=4)

    parser.add_argument('--poison-path', type=str, default=None, help='the path of pretrained poison')
    parser.add_argument('--watermark-path', type=str, default=None, help='the path of watermarks')
    parser.add_argument('--poison-size', type=int, default=32,
                        help='the image size of poisons')

    parser.add_argument('--optimizer', default='sgd', type=str,
                        help='the optimizer used in training')
    parser.add_argument('--epochs', default=200, type=int,
                        help='the number of total epochs to run')
    parser.add_argument('--lr', default=0.5, type=float, help='optimizer learning rate')
    parser.add_argument('--seed', default=None, type=int, help='random seed')

    parser.add_argument('--resume', action='store_true', help='if resume training')
    parser.add_argument('--cutout', action='store_true', help='use cutout')
    parser.add_argument('--cutmix', action='store_true', help='use cutmix')
    parser.add_argument('--mixup', action='store_true', help='use mixup')
    parser.add_argument('--gaussian-smooth', action='store_true', help='if use gaussian smooth')
    parser.add_argument('--random-noise', action='store_true', help='if use random noise')
    parser.add_argument('--gpu-id', type=str, default='0', help='the gpu id')

    parser.add_argument('--poisoned-class', default=-1, type=int,
                        help='which class could be poisoned, if all classes are poisoned, it is set to -1')

    parser.add_argument('--get-lr-process', action='store_true', help='if get learning process')

    parser.add_argument('--post-poisoning', action='store_true',
                        help='if generate post-poisoning watermark')
    parser.add_argument('--mask-type', default='fixed', choices=['random', 'fixed', 'fix-lt',
                                                                 'fix-lb', 'fix-rt', 'fix-rb'],
                        help='the type of mask for pixels')

    parser.add_argument('--targeted-class', default=2, type=int,
                        help='which class could be as target class')

    parser.add_argument('--watermark-budget', type=float, default=8, help='the watermark budget')

    parser.add_argument('--poison-ratio', type=float, default=0.01, help='the poison ratio for one class')
    parser.add_argument('--multi-test', type=float, default=3.0, help='the scale of test trigger')
    parser.add_argument('--wm-length', type=int, default=2000, help='the watermarking length')

    arguments = parser.parse_args()
    arguments.watermark_budget = arguments.watermark_budget / 255

    return arguments
def loss_mix(y, logits):
    criterion = F.cross_entropy
    loss_a = criterion(logits, y[:, 0].long(), reduction="none")
    loss_b = criterion(logits, y[:, 1].long(), reduction="none")
    return ((1 - y[:, 2]) * loss_a + y[:, 2] * loss_b).mean()

def train_epoch(train_loader, model, optimizer, scheduler, epoch, log):
    losses = AverageMeter()
    data_time_meter = AverageMeter()
    train_time_meter = AverageMeter()
    current_lr = optimizer.state_dict()['param_groups'][0]['lr']
    start = time.time()
    if args.cutmix:
        cutmix = K.RandomCutMixV2(data_keys=["input", "class"])
    elif args.mixup:
        mixup = K.RandomMixUpV2(data_keys=["input", "class"])

    for i, (data, target, _) in enumerate(train_loader):
        data = data.cuda()
        #print(target)
        target = target.cuda()
        data_time = time.time() - start
        data_time_meter.update(data_time)

        if args.cutmix or args.mixup:
            if args.cutmix:
                data, target = cutmix(data, target)
                target = target.squeeze(0)
            elif args.mixup:
                data, target = mixup(data, target)
            #print(target)
            features = model.train()(data)
            loss = loss_mix(target, features)
        else:
            features = model.train()(data)
            loss = F.cross_entropy(features, target)
        optimizer.zero_grad()
        loss.backward()
        optimizer.step()
        losses.update(loss.item(), data.shape[0])

        train_time = time.time() - start
        train_time_meter.update(train_time)
        start = time.time()
    log.info(
        f'TRAINING\t'
        f'Epoch[{epoch}/{args.epochs}]\t'
        f'avg loss = {losses.avg:.4f}\t'
        f'epoch time = {train_time_meter.sum:.2f}\t'
        f'data time = {data_time_meter.sum:.2f}\t'
        f'current lr = {current_lr:.4f}'
    )
    scheduler.step()


def main():
    if args.seed is not None:
        setup_seed(args.seed)
    os.environ['CUDA_VISIBLE_DEVICES'] = args.gpu_id
    if args.post_poisoning:
        save_dir = os.path.join('eval', args.dataset, 'post_poisoning')
    else:
        save_dir = os.path.join('eval', args.dataset, 'poisoning_concurrent')
    if not os.path.exists(save_dir):
        os.makedirs(save_dir)
    log = logger(path=save_dir)
    log.info(str(args))

    try:
        poison = torch.load(args.poison_path, map_location='cpu')
        #poison = torch.clamp(poison, -args.poison_budget, args.poison_budget)
        print('poison:', torch.max(poison))
        print('use loaded noise.')
    except:
        if args.poison_path == 'random':
            torch.manual_seed(1)
            poison = torch.randn([50000, 3, 32, 32]).uniform_(-8/255, 8/255)
            print('use random noise.')
            print('poison:', poison)
        else:
            poison = None
            print('no poison founded!')

    if args.post_poisoning:
        if args.mask_type == 'random':
            C, H, W = 3, args.poison_size, args.poison_size
            mask = torch.zeros(1, C, H, W).cuda()

            num_pixels = C * H * W  # Total number of pixels in C * H * W dimensions
            num_unmasked = int(num_pixels - args.wm_length)  # Number of unmasked pixels

            # For each batch, randomly select the initial unmasked pixels based on C * H * W
            flat_mask = mask[0].view(-1)  # Flatten the C * H * W part of the mask
            unmasked_indices = torch.randperm(num_pixels)[:num_unmasked]  # Initial random unmasking
            flat_mask[unmasked_indices] = 1  # Set the selected pixels to be unmasked (value = 1)

            # Broadcast the mask to all channels (C), so it has the shape 1 x C x H x W
            mask = flat_mask.view(1, C, H, W).cuda()


        elif args.mask_type == 'fixed':
            C, H, W = 3, args.poison_size, args.poison_size
            mask = torch.zeros(1, C, H, W)

            num_pixels = C * H * W  # Total number of pixels in C * H * W dimensions

            # rat = np.sqrt(1.0 - args.wm_rate)
            rat = np.sqrt(1.0 - (args.wm_length / num_pixels))

            for c in range(C):
                for h in range(int(rat * H)):
                    for w in range(int(rat * W)):
                        mask[0, c, h, w] = 1

            mask = mask.cuda()

        elif args.mask_type == 'fix-lt':
            C, H, W = 3, args.poison_size, args.poison_size
            mask = torch.ones(1, C, H, W)
            num_pixels = C * H * W  # Total number of pixels in C * H * W dimensions
            rat = np.sqrt(args.wm_length / num_pixels)

            for c in range(C):
                for h in range(int(rat * H)):
                    for w in range(int(rat * W)):
                        mask[0, c, h, w] = 0

            mask = mask.cuda()

        elif args.mask_type == 'fix-lb':
            C, H, W = 3, args.poison_size, args.poison_size
            mask = torch.ones(1, C, H, W)
            num_pixels = C * H * W  # Total number of pixels in C * H * W dimensions
            rat = np.sqrt(args.wm_length / num_pixels)

            for c in range(C):
                for h in range(H - int(rat * H), H):
                    for w in range(int(rat * W)):
                        mask[0, c, h, w] = 0

            mask = mask.cuda()

        elif args.mask_type == 'fix-rt':
            C, H, W = 3, args.poison_size, args.poison_size
            mask = torch.ones(1, C, H, W)
            num_pixels = C * H * W  # Total number of pixels in C * H * W dimensions
            rat = np.sqrt(args.wm_length / num_pixels)

            for c in range(C):
                for h in range(int(rat * H)):
                    for w in range(W - int(rat * W), W):
                        mask[0, c, h, w] = 0

            mask = mask.cuda()

        elif args.mask_type == 'fix-rb':
            C, H, W = 3, args.poison_size, args.poison_size
            mask = torch.ones(1, C, H, W)
            num_pixels = C * H * W  # Total number of pixels in C * H * W dimensions
            rat = np.sqrt(args.wm_length / num_pixels)

            for c in range(C):
                for h in range(H - int(rat * H), H):
                    for w in range(W - int(rat * W), W):
                        mask[0, c, h, w] = 0

            mask = mask.cuda()

        else:
            C, H, W = 3, args.poison_size, args.poison_size
            mask = torch.ones(1, C, H, W).cuda()

        watermark = torch.where(torch.randn(1, C, H, W) < 0, -args.watermark_budget, args.watermark_budget) * (
                torch.ones(1, C, H, W) - mask.cpu())

    else:
        watermark = torch.load(args.watermark_path, map_location='cpu')



    transform = transforms.Compose([transforms.RandomCrop(32, 4),
                        transforms.RandomHorizontalFlip(p=0.5),
                        transforms.ToTensor()])

    #poison = torch.zeros(50000, 3, 32, 32)
    #targeted = args.target_class
    poison = poison.view(3,32,32)
    print(poison.size())
    multi_test = args.multi_test
    #if args.post_poisoning:
    train_poison = poison + watermark.view(3,32,32)
    #else:
    #    train_poison = poison
    train_set = CIFAR10Index(root=args.data, train=True, transform=transform, delta=train_poison, download=True, poisoned_class=args.targeted_class, ratio=args.poison_ratio)
    clean_test_set = CIFAR10Index(root=args.data, train=False, transform=transform, delta=None, download=True, poisoned_class=args.targeted_class)
    backdoor_test_set = CIFAR10Index(root=args.data, train=False, transform=transform, delta=poison*multi_test, download=True, trigger=True, poisoned_class=args.targeted_class)
    train_loader = DataLoader(train_set, batch_size=args.batch_size, shuffle=True)
    clean_test_loader = DataLoader(clean_test_set, batch_size=args.batch_size, shuffle=False)
    backdoor_test_loader = DataLoader(backdoor_test_set, batch_size=args.batch_size, shuffle=False)


    im100 = True if args.dataset == 'imagenet100' else False
    if args.backbone == 'resnet18':
        model = resnet18(num_classes=args.num_classes).cuda()
        if args.dataset in ['cifar10', 'cifar100']:
            model.conv1 = nn.Conv2d(3, 64, 3, 1, 1, bias=False).cuda()
            model.maxpool = nn.Identity().cuda()
    elif args.backbone == 'resnet50':
        model = resnet50(num_classes=args.num_classes).cuda()
        if args.dataset in ['cifar10', 'cifar100']:
            model.conv1 = nn.Conv2d(3, 64, 3, 1, 1, bias=False).cuda()
            model.maxpool = nn.Identity().cuda()
    else:
        raise AssertionError('model is not defined')

    if args.dataset == 'imagenet100':
        model = nn.DataParallel(model)

    if args.optimizer == 'adam':
        optimizer = torch.optim.Adam(model.parameters(), lr=args.lr)
    elif args.optimizer == 'sgd':
        optimizer = torch.optim.SGD(model.parameters(), lr=args.lr, weight_decay=1e-4, momentum=0.9)
    else:
        raise AssertionError('optimizer is not defined')

    scheduler = torch.optim.lr_scheduler.LambdaLR(
        optimizer,
        lr_lambda=lambda step: cosine_annealing(step,
                                                args.epochs,
                                                1,
                                                1e-6 / args.lr,
                                                warmup_steps=0)
    )

    start_epoch = 1
    if args.resume:
        checkpoint = torch.load(os.path.join(save_dir, 'model.pt'))
        start_epoch = checkpoint['epoch'] + 1
        model.load_state_dict(checkpoint['state_dict'])
        optimizer.load_state_dict(checkpoint['optim'])
        for i in range(start_epoch - 1):
            scheduler.step()
        log.info(f"RESUME FROM EPOCH {start_epoch-1}")

    if args.cutmix:
        cutmix = K.RandomCutMixV2(data_keys=["input", "class"])

    if args.get_lr_process:
        val_acc_record = []
        test_acc_record = []
        asr_record = []

    for epoch in range(start_epoch, args.epochs + 1):
        train_epoch(train_loader, model, optimizer, scheduler, epoch, log)

        if args.get_lr_process:

            save_checkpoint({
                'epoch': epoch,
                'state_dict': model.state_dict(),
                'optim': optimizer.state_dict(),
            }, filename=os.path.join(save_dir, 'model.pt'))
            val_acc = evaluation(train_loader, model)
            clean_test_acc = evaluation(clean_test_loader, model)
            attack_success_rate = evaluation(backdoor_test_loader, model)
            val_acc_record.append(val_acc)
            test_acc_record.append(clean_test_acc)
            asr_record.append(attack_success_rate)
            print(val_acc_record, test_acc_record)
            log.info(
                f'val accuracy = {val_acc:.4f}\t'
                f'test accuracy = {clean_test_acc:.4f}'
                f'ASR = {attack_success_rate:.4f}'
            )

        else:
            if epoch % 10 == 0:
                save_checkpoint({
                    'epoch': epoch,
                    'state_dict': model.state_dict(),
                    'optim': optimizer.state_dict(),
                }, filename=os.path.join(save_dir, 'model.pt'))

                val_acc = evaluation(train_loader, model)
                test_acc = evaluation(clean_test_loader, model)
                attack_success_rate = evaluation(backdoor_test_loader, model)
                log.info(
                    f'val accuracy = {val_acc:.4f}\t'
                    f'test accuracy = {test_acc:.4f}\t'
                    f'ASR = {attack_success_rate:.4f}'
                )
    if args.get_lr_process:
        log.info(
            f'val accuracy record = {val_acc_record}\t'
            f'test accuracy record = {test_acc_record}\t'
            f'ASR record = {asr_record}'
        )


if __name__ == '__main__':
    args = get_args()
    if args.dataset == 'cifar10':
        args.num_classes = 10
        args.poison_size = 32
    if args.dataset == 'cifar100':
        args.num_classes = 100
        args.poison_size = 32
    if args.dataset == 'tinyimagenet':
        args.num_classes = 200
        args.poison_size = 64
    if args.dataset == 'miniimagenet':
        args.num_classes = 100
        args.poison_size = 84
    if args.dataset == 'imagenet100':
        args.num_classes = 100
        args.poison_size = 224

    cudnn.benchmark = True
    main()
