from collections import OrderedDict

import numpy as np
import wandb
import torch
import torch.nn as nn
import torch.optim as optim
import torch.autograd as autograd
import torch.distributions
from torch.optim.lr_scheduler import StepLR

from .base_agent import BaseAgent
from .base_il_agent import BaseILAgent
from .ppo_agent import PPOAgent
from .dataset import ReplayBuffer, RandomSampler
from .expert_dataset import ExpertDataset
from networks.discriminator import Discriminator
from utils.info_dict import Info
from utils.logger import logger
from utils.mpi import mpi_average
from utils.pytorch import (
    optimizer_cuda,
    count_parameters,
    sync_networks,
    sync_grads,
    to_tensor,
    center_crop_images,
)
import copy
import numpy as np

class GAILAgent(BaseILAgent):
    def __init__(self, config, ob_space, ac_space, env_ob_space, layout):
        super().__init__(config, ob_space, ac_space, env_ob_space, layout)

        self._rl_agent.set_reward_function(self._predict_reward)

        self.encoder = None
        self._discriminator = Discriminator(
            config, ob_space, ac_space if not config.gail_no_action else None, self.encoder
        )
        self._discriminator_loss = nn.BCEWithLogitsLoss()
        self._network_cuda(config.device)

        # build optimizers
        self._discriminator_optim = optim.Adam(
            self._discriminator.parameters(), lr=config.discriminator_lr
        )

        # build learning rate scheduler
        self._discriminator_lr_scheduler = StepLR(
            self._discriminator_optim,
            step_size=(self._config.max_global_step // self._config.rollout_length // 5) if config.gail_rl_algo=='ppo' else (self._config.max_global_step // 5),
            gamma=0.5,
        )

        # expert dataset
        if config.is_train and config.bc_loss_coeff > 0.0:
            self._rl_agent.set_bc_data_iter(self._data_iter)
            self._rl_agent.set_bc_data_loader(self._data_loader)


        # load other expert demos
        if config.pretrain_discriminator or config.blend_ratio!=0.0:
            self._other_dataset = ExpertDataset(
                config.other_demo_path,
                config.demo_subsample_interval,
                ac_space,
                use_low_level=config.demo_low_level,
                sample_range_start=config.demo_sample_range_start,
                sample_range_end=config.demo_sample_range_end,
                num_task=config.num_task,
                target_taskID=None,
                with_taskID= config.with_taskID,
            )
            self._other_data_loader = torch.utils.data.DataLoader(
                self._other_dataset,
                batch_size=self._config.batch_size,
                shuffle=True,
                drop_last=True,
            )
            self._other_data_iter = iter(self._other_data_loader)


        # policy dataset
        sampler = RandomSampler(image_crop_size = config.encoder_image_size)
        if config.gail_rl_algo=='ppo':
            keys = ["ob", "ob_next", "ac", "done", "rew", "ret", "adv", "ac_before_activation", "success", "traj_start_indicator"]
        else:
            keys = ["ob", "ob_next", "ac", "done", "done_mask", "rew"]

        if self._config.with_taskID:
            keys.append("id")
        
        self._buffer = ReplayBuffer(
            keys,
            config.rollout_length if config.gail_rl_algo=='ppo' else config.buffer_size,
            sampler.sample_func,
        )

        self._rl_agent.set_buffer(self._buffer)
        self._update_iter = 0
        if self._config.gail_rl_algo=='ppo':
            self._num_batches = (  # = 3
                        config.rollout_length
                        // config.batch_size
                        // config.discriminator_update_freq
                )
            assert self._num_batches > 0
        # update observation normalizer with dataset
        self.update_normalizer()

        self._log_creation()

    def pretrain_dis(self):
        num_iter = len(self._other_dataset) // self._config.batch_size
        
        def _train_for_other_task():
            num_file = self._other_dataset.num_files # = 3
            one_hot_encode = self._other_dataset.one_hot_encode
            for i in range(num_file):
                target_taskID = np.array(one_hot_encode[i]).reshape(1, -1)
                task_id = torch.as_tensor(target_taskID)

                for s_ in range(num_iter*num_file):
                    try:
                        other_expert_data = next(self._other_data_iter)
                    except StopIteration:
                        self._other_data_iter = iter(self._other_data_loader)
                        other_expert_data = next(self._other_data_iter)
                             
                    target_data, other_data = {}, {}
                    current_data_indx = (other_expert_data["id"]==task_id).all(1)
                    other_data_index = ~current_data_indx

                    for k, v in other_expert_data.items():
                        if isinstance(v, dict):
                            sub_keys = v.keys()
                            target_data[k] = {sub_key: v[sub_key][current_data_indx] for sub_key in sub_keys}
                            other_data[k] = {sub_key: v[sub_key][other_data_index] for sub_key in sub_keys}
                        else:
                            target_data[k] = v[current_data_indx]
                            other_data[k] = v[other_data_index]

                    len_current_data, len_other_data = sum(current_data_indx), sum(other_data_index)
                    if len_current_data != len_other_data:
                        bs = min(len_current_data, len_other_data)
                        if len_current_data > len_other_data:
                            for k, v in target_data.items():
                                if isinstance(v, dict):
                                    sub_keys = v.keys()
                                    target_data[k] = {sub_key: v[sub_key][:bs] for sub_key in sub_keys}
                                else:
                                    target_data[k] = v[:bs]
                        else:
                            for k, v in other_data.items():
                                if isinstance(v, dict):
                                    sub_keys = v.keys()
                                    other_data[k] = {sub_key: v[sub_key][:bs] for sub_key in sub_keys}
                                else:
                                    other_data[k] = v[:bs]

                    _train_info = self._update_discriminator(other_data, target_data, pretrain_discriminator=True)
                    for k_, v_ in _train_info.items():
                        if np.isscalar(v_) or (hasattr(v_, "shape") and np.prod(v_.shape) == 1):
                            wandb.log({"pretrain_other_task_dis/%s" % k_: v_}, step=s_)
            logger.info("Pretrain other tasks finished (first method)")

        if self._config.pretrain_discriminator == 1:
            _train_for_other_task()
        
        for s_ in range(num_iter):
            try:
                expert_data = next(self._data_iter)
            except StopIteration:
                self._data_iter = iter(self._data_loader)
                expert_data = next(self._data_iter)

            try:
                other_expert_data = next(self._other_data_iter)
            except StopIteration:
                self._other_data_iter = iter(self._other_data_loader)
                other_expert_data = next(self._other_data_iter)

            _train_info = self._update_discriminator(other_expert_data, expert_data, pretrain_discriminator=True)

            for k, v in _train_info.items():
                if np.isscalar(v) or (hasattr(v, "shape") and np.prod(v.shape) == 1):
                    wandb.log({"pretrain_dis/%s" % k: v}, step=s_)

        logger.info("Pretrain discriminator finished") 

    def _predict_reward(self, ob, ac, ID=None, *argv):
        if self._config.gail_no_action:
            ac = None
        elif self._discrete:
            ac = self._ac_one_hot(ac)
        with torch.no_grad():
            ret, _ = self._discriminator(ob, ac, ID)
            eps = 1e-10
            s = torch.sigmoid(ret)
            if self._config.gail_reward == "vanilla":
                reward = -(1 - s + eps).log()
            elif self._config.gail_reward == "gan":
                reward = (s + eps).log() - (1 - s + eps).log()
            elif self._config.gail_reward == "d":
                reward = ret

        reward_dict = {"rew": reward, "gail_rew": reward}
        return reward_dict

    def predict_reward(self, ob, ac=None, ID=None, *argv):
        ob = self.normalize(ob)
        ob = to_tensor(ob, self._config.device)
        if self._config.gail_no_action:
            ac = None
        if ac is not None:
            ac = to_tensor(ac, self._config.device)
        if ID is not None:
            ID = to_tensor(ID, self._config.device)

        reward_dict = self._predict_reward(ob, ac, ID, argv)
        # if self._config.one_hot_vector:
        #     for k, v in reward_dict.items():
        #         reward_dict[k] = v[:, 3:4].cpu().item()
        # else:
        for k, v in reward_dict.items():
            if torch.numel(v) == 1:
                reward_dict[k] = v.cpu().item()
            else:
                reward_dict[k] = v.cpu().numpy()
        return reward_dict

    def _log_creation(self):
        if self._config.is_chef:
            logger.info("Creating a GAIL agent")
            logger.info(
                "The discriminator has %d parameters",
                count_parameters(self._discriminator),
            )

    def state_dict(self):
        ckpt = super().state_dict()
        ckpt.update({
            "discriminator_state_dict": self._discriminator.state_dict(),
            "discriminator_optim_state_dict": self._discriminator_optim.state_dict(),
            "ob_norm_state_dict": self._ob_norm.state_dict(),
        })
        return ckpt

    def load_state_dict(self, ckpt):
        super().load_state_dict(ckpt)

        if "rl_agent" in ckpt:
            self._discriminator.load_state_dict(ckpt["discriminator_state_dict"])
            self._ob_norm.load_state_dict(ckpt["ob_norm_state_dict"])
            self._network_cuda(self._config.device)

            self._discriminator_optim.load_state_dict(
                ckpt["discriminator_optim_state_dict"]
            )
            optimizer_cuda(self._discriminator_optim, self._config.device)

    def _network_cuda(self, device):
        self._discriminator.to(device)

    def sync_networks(self):
        super().sync_networks()
        sync_networks(self._discriminator)

    def update_normalizer(self, obs=None):
        """ Updates normalizers for discriminator and PPO agent. """
        if self._config.ob_norm:
            if obs is None:
                data_loader = torch.utils.data.DataLoader(
                    self._dataset,
                    batch_size=self._config.batch_size,
                    shuffle=False,
                    drop_last=False,
                )
                for obs in data_loader:
                    super().update_normalizer(obs)
                    self._rl_agent.update_normalizer(obs)
            else:
                super().update_normalizer(obs)
                self._rl_agent.update_normalizer(obs)

    def _blend(self, policy_data, other_expert_data, blend_ratio):
        """ Blend expert and policy data. """
        other_demo_num = int(self._config.batch_size * blend_ratio)
        new_transitions = {}
        for k, v in other_expert_data.items():
            if isinstance(v, dict):
                sub_keys = v.keys()
                new_transitions[k] = {
                    sub_key: np.concatenate((policy_data[k][sub_key][:other_demo_num], v[sub_key][other_demo_num:])) for sub_key in sub_keys
                }
            else:
                new_transitions[k] = np.concatenate((policy_data[k][:other_demo_num], v[other_demo_num:]))

        return new_transitions

    def train(self, step=0):
        train_info = Info()

        self._discriminator_lr_scheduler.step()

        if self._config.gail_rl_algo!='ppo':
            self._num_batches = (self._update_iter % self._config.discriminator_update_freq == 0)
            self._update_iter += 1
        
        total_num_blend = self._config.warm_up_steps+self._config.blend_steps
        half_num_blend = self._config.warm_up_steps+self._config.blend_steps/2
        
        for _ in range(self._num_batches):
            policy_data = self._buffer.sample(self._config.batch_size)

            try:
                expert_data = next(self._data_iter)
            except StopIteration:
                self._data_iter = iter(self._data_loader)
                expert_data = next(self._data_iter)
            
            if self._config.blend_ratio != 0.0 and step < total_num_blend:
                # add other demos to policy samples
                policy_data_deepcopy = copy.deepcopy(policy_data)
                if step < half_num_blend:
                    blend_ratio = self._config.blend_ratio
                else:
                    blend_ratio = 0.9
                
                try:
                    other_expert_data = next(self._other_data_iter)
                except StopIteration:
                    self._other_data_iter = iter(self._other_data_loader)
                    other_expert_data = next(self._other_data_iter)
                #blend other demos with policy samples
                policy_data_deepcopy = self._blend(policy_data_deepcopy, other_expert_data, blend_ratio)
            
                _train_info = self._update_discriminator(policy_data_deepcopy, expert_data, step=step)
            else:
                _train_info = self._update_discriminator(policy_data, expert_data, step=step)
            train_info.add(_train_info)

        _train_info = self._rl_agent.train()
        train_info.add(_train_info)

        for _ in range(self._num_batches):
            try:
                expert_data = next(self._data_iter)
            except StopIteration:
                self._data_iter = iter(self._data_loader)
                expert_data = next(self._data_iter)
            self.update_normalizer(expert_data["ob"])

        return train_info.get_dict(only_scalar=True)

    def _update_discriminator(self, policy_data, expert_data, pretrain_discriminator=False, step=0):
        info = Info()

        _to_tensor = lambda x: to_tensor(x, self._config.device)
        # pre-process observations
        p_o = policy_data["ob"]
        p_o = self.normalize(p_o)
        p_o = _to_tensor(p_o)

        e_o = expert_data["ob"]
        e_o = self.normalize(e_o)
        e_o = _to_tensor(e_o)
        # if self._config.encoder_type == "cnn":
        #     # change e_o to channel first
        #     e_o = {"ob": e_o}
        #     for k, v in e_o.items():
        #         if self._config.encoder_type == "cnn" and len(v['ob'].shape) == 4:
        #             e_o[k] = center_crop_images(v['ob'], self._config.encoder_image_size)


        if self._config.gail_no_action:
            p_ac = None
            e_ac = None
        else:
            p_ac = _to_tensor(policy_data["ac"])
            e_ac = _to_tensor(expert_data["ac"])

            # if self._discrete:
            #     p_ac = self._ac_one_hot(p_ac)
            #     e_ac = self._ac_one_hot(e_ac)

        # else:
        p_id = None
        e_id = None
        

        p_logit, p_z = self._discriminator(p_o, p_ac, p_id)
        e_logit, e_z = self._discriminator(e_o, e_ac, e_id)

        p_output = torch.sigmoid(p_logit)
        e_output = torch.sigmoid(e_logit)

        p_loss = self._discriminator_loss(
            p_logit, torch.zeros_like(p_logit).to(self._config.device)
        )
        e_loss = self._discriminator_loss(
            e_logit, torch.ones_like(e_logit).to(self._config.device)
        )

        logits = torch.cat([p_logit, e_logit], dim=0)
        entropy = torch.distributions.Bernoulli(logits=logits).entropy().mean()
        entropy_loss = -self._config.gail_entropy_loss_coeff * entropy

        grad_pen = self._compute_grad_pen(p_o, p_ac, e_o, e_ac, p_id, pretrain_discriminator)
        grad_pen_loss = self._config.gail_grad_penalty_coeff * grad_pen

        gail_loss = p_loss + e_loss + entropy_loss + grad_pen_loss

        # update the discriminator
        self._discriminator.zero_grad()
        gail_loss.backward()
        sync_grads(self._discriminator)
        self._discriminator_optim.step()

        info["gail_neg_output"] = p_output.mean().detach().cpu().item()
        info["gail_pos_output"] = e_output.mean().detach().cpu().item()
        info["gail_entropy"] = entropy.detach().cpu().item()
        info["gail_neg_loss"] = p_loss.detach().cpu().item()
        info["gail_pos_loss"] = e_loss.detach().cpu().item()
        info["gail_entropy_loss"] = entropy_loss.detach().cpu().item()
        info["gail_grad_pen"] = grad_pen.detach().cpu().item()
        info["gail_grad_loss"] = grad_pen_loss.detach().cpu().item()
        info["gail_loss"] = gail_loss.detach().cpu().item()

        return mpi_average(info.get_dict(only_scalar=True))

    def _compute_grad_pen(self, policy_ob, policy_ac, expert_ob, expert_ac, policy_id=None, pretrain_discriminator=False):
        if pretrain_discriminator:
            if isinstance(policy_ob, OrderedDict) or isinstance(policy_ob, dict):
                batch_size = list(policy_ob.values())[0].shape[0]
            else:
                batch_size = policy_ob[0].shape[0]
        else:
            batch_size = self._config.batch_size
        alpha = torch.rand(batch_size, 1, device=self._config.device)

        def blend_dict(a, b, alpha):
            if isinstance(a, dict):
                return OrderedDict(
                    [(k, blend_dict(a[k], b[k], alpha)) for k in a.keys()]
                )
            elif isinstance(a, list):
                return [blend_dict(a[i], b[i], alpha) for i in range(len(a))]
            else:
                if self._config.encoder_type == "cnn":
                    if len(a.shape) == 4 and len(alpha.shape) != 4:
                        alpha = alpha.unsqueeze_(-1).unsqueeze_(-1)
                    if len(a.shape) == 2 and len(alpha.shape) != 2:
                        alpha = alpha.squeeze(-1).squeeze(-1)
                    
                expanded_alpha = alpha.expand_as(a)
                ret = expanded_alpha * a + (1 - expanded_alpha) * b
                ret.requires_grad = True
                return ret

        interpolated_ob = blend_dict(policy_ob, expert_ob, alpha)
        inputs = list(interpolated_ob.values())
        if policy_ac is not None:
            interpolated_ac = blend_dict(policy_ac, expert_ac, alpha)
            inputs = inputs + list(interpolated_ob.values())
        else:
            interpolated_ac = None

        interpolated_logit, _ = self._discriminator(interpolated_ob, interpolated_ac, policy_id)
        ones = torch.ones(interpolated_logit.size(), device=self._config.device)

        # if policy_id is not None:
        #     policy_id.requires_grad_()
        #     inputs = inputs + list(policy_id)

        grad = autograd.grad(
            outputs=interpolated_logit,
            inputs=inputs,
            grad_outputs=ones,
            create_graph=True,
            retain_graph=True,
            only_inputs=True,
        )[0]
        #grad = grad[0]

        grad_pen = (grad.norm(2, dim=1) - 1).pow(2).mean()
        return grad_pen
