# CACTUS: Compression Aware Certified Training Using network Sets

This repository contains the implementation of **CACTUS**, a unified framework for training neural networks that remain certifiably robust even after compression. Our method jointly optimizes for accuracy, certified robustness, and compressibility during training.

## Overview

CACTUS addresses the critical gap between neural network compression and certified robustness. Traditional approaches treat these as separate objectives, forcing practitioners to choose between efficiency and safety. CACTUS enables models to maintain high certified accuracy even when compressed through pruning or quantization.

## Key Features

- **Unified Training**: Joint optimization of compression and certified robustness
- **Multiple Compression Types**: Support for both pruning and quantization
- **Flexible Framework**: Works with different certified training methods (SABR, IBP)
- **Proven Effectiveness**: State-of-the-art results on MNIST and CIFAR-10

## Installation

```bash
# Clone the repository
git clone <repository-url>
cd cactus

# Create conda environment
conda create -n cactus python=3.8
conda activate cactus

# Install dependencies
pip install -r requirements.txt
```

## Repository Structure

```
cactus/
├── src/
│   ├── models/           # Neural network architectures
│   ├── compression/      # Pruning and quantization methods
│   ├── training/         # CACTUS training algorithms
│   ├── certification/    # Certified training losses (SABR, IBP)
│   ├── baselines/        # Baseline methods (HYDRA, NRSLoss, QA-IBP)
│   └── utils/           # Utility functions
├── experiments/         # Experiment scripts and configs
├── data/               # Dataset loading and preprocessing
├── results/            # Experimental results and plots
└── tests/              # Unit tests
```

## Quick Start

### Training CACTUS Model

```bash
# Train CACTUS with pruning on MNIST
python experiments/train_cactus.py --dataset mnist --compression pruning --epsilon 0.1

# Train CACTUS with quantization on CIFAR-10
python experiments/train_cactus.py --dataset cifar10 --compression quantization --epsilon 2/255
```

### Running Evaluations

```bash
# Evaluate on pruning benchmarks
python experiments/evaluate_pruning.py --model_path results/cactus_mnist.pth

# Evaluate on quantization benchmarks
python experiments/evaluate_quantization.py --model_path results/cactus_cifar10.pth
```

### Reproducing Paper Results

```bash
# Run all experiments from the paper
bash experiments/reproduce_results.sh
```

## Method Overview

CACTUS optimizes the following joint objective:

```
L_CACTUS = (1/|C(f_θ)|) Σ [λ * L_std(C_ψ(x), y) + (1-λ) * L_cert(C_ψ(x), y)]
```

Where:
- `C(f_θ)` is the compression set containing networks at different compression levels
- `C_ψ(x)` represents compressed networks with parameterization ψ
- `L_std` is standard cross-entropy loss
- `L_cert` is certified loss (e.g., SABR)

## Key Components

### Compression Methods
- **Pruning**: Global/local, structured/unstructured L1/L2 pruning
- **Quantization**: FP16, INT8 with Adversarial Weight Perturbation (AWP)

### Certified Training
- **SABR**: Smart Adversarial Boundary Regularization
- **IBP**: Interval Bound Propagation
- **Auto-LiRPA**: Integration for bound computation

### Model Architectures
- **CNN7**: 7-layer CNN used in experiments
- **Support for**: ResNet, VGG architectures