import torch
import torchinfo
import torch.nn as nn
from spikingjelly.clock_driven.neuron import (
    MultiStepParametricLIFNode,
    MultiStepLIFNode,
)
from spikingjelly.clock_driven import layer
from timm.models.layers import to_2tuple, trunc_normal_, DropPath
from timm.models.registry import register_model
from timm.models.vision_transformer import _cfg
from einops.layers.torch import Rearrange
import torch.nn.functional as F
from functools import partial
from spikingjelly.clock_driven import surrogate
from timm.models import create_model, safe_model_name, resume_checkpoint, load_checkpoint, \
    convert_splitbn_model, model_parameters
from functools import partial
from abc import abstractmethod
__all__ = ['spikformer']
# CUDA_VISIBLE_DEVICES=0 python -m torch.distributed.launch --nproc_per_node=1 --master_port 29501 train.py -c cifar10.yml 
class DownSampling(nn.Module):
    def __init__(
        self,
        in_channel,
        out_channel,
        first_layer=False
    ):
        super().__init__()
        
        self.conv = nn.Conv2d(in_channel, out_channel, kernel_size=7, stride=2, padding=3)
        self.dconv = nn.Conv2d(in_channel, out_channel, kernel_size=3, dilation=2, padding=2, stride=2)

        self.bn = nn.BatchNorm2d(out_channel)
        self.dbn = nn.BatchNorm2d(out_channel)

        self.fc = nn.Conv2d(in_channel, out_channel, kernel_size=1, stride=1)
        self.fbn = nn.BatchNorm2d(out_channel)
        self.pool = nn.MaxPool2d((2, 2))
        self.lif = MultiStepLIFNode(tau=2.0, detach_reset=True, backend="cupy")
        self.first_layer = first_layer

    def forward(self, x):

        T, B, C, H, W = x.shape
        
        if self.first_layer==False:
            x = self.lif(x)
        
        x_conv = self.bn(self.conv(x.flatten(0, 1)))
        x_dconv = self.dbn(self.dconv(x.flatten(0, 1)))
        x_fc = self.pool(self.fbn(self.fc(x.flatten(0, 1))))

        x = x_conv + x_dconv + x_fc

        x = x.reshape(T, B, -1, H // 2, W // 2)

        return x

class GL_Conv(nn.Module):

    def __init__(self, channels):
        super().__init__()

        self.conv = nn.Conv2d(channels, channels, kernel_size=7, stride=1, padding=3)
        self.bn = nn.BatchNorm2d(channels)

        self.dconv = nn.Conv2d(channels, channels, kernel_size=3, dilation=2, padding=2, stride=1)
        self.dbn = nn.BatchNorm2d(channels)

        self.lif = MultiStepLIFNode(tau=2.0, detach_reset=True, backend="cupy")

    def forward(self, x):
        T, B, C, H, W = x.shape
        
        x_conv = self.bn(self.conv(x.flatten(0, 1)))

        x_dconv = self.dbn(self.dconv(x.flatten(0, 1)))

        x = (x_conv + x_dconv).reshape(T, B, -1, H, W)

        return x

class MS_Attention(nn.Module):
    def __init__(
        self,
        dim,
        num_heads=8,
    ):
        super().__init__()
        assert (
            dim % num_heads == 0
        ), f"dim {dim} should be divided by num_heads {num_heads}."
        self.dim = dim
        self.num_heads = num_heads
        self.scale = 0.125

        self.lif = MultiStepLIFNode(tau=2.0, detach_reset=True, backend="cupy")

        self.q_conv = GL_Conv(dim)

        self.q_lif = MultiStepLIFNode(tau=2.0, detach_reset=True, backend="cupy")

        self.k_conv = GL_Conv(dim)

        self.k_lif = MultiStepLIFNode(tau=2.0, detach_reset=True, backend="cupy")

        self.v_conv = GL_Conv(dim)

        self.v_lif = MultiStepLIFNode(tau=2.0, detach_reset=True, backend="cupy")

        self.proj_conv = GL_Conv(dim)


        self.fc = nn.Linear(4, 4)
        nn.init.constant_(self.fc.bias, -0.5)

        self.surrogate_function = surrogate.ATan()

    def forward(self, x):

        T, B, C, H, W = x.shape
        N = H * W

        x_clone = x.clone()
        
        x = self.lif(x)

        q = self.q_conv(x).reshape(T, B, C, H, W)
        k = self.k_conv(x).reshape(T, B, C, H, W)
        v = self.v_conv(x).reshape(T, B, C, H, W)

        q = self.q_lif(q).flatten(3)
        q = (
            q.transpose(-1, -2)
            .reshape(T, B, N, self.num_heads, C // self.num_heads)
            .permute(0, 1, 3, 2, 4)
            .contiguous()
        )

        k = self.k_lif(k).flatten(3)
        k = (
            k.transpose(-1, -2)
            .reshape(T, B, N, self.num_heads, C // self.num_heads)
            .permute(0, 1, 3, 2, 4)
            .contiguous()
        ) # T, B, self.num_heads, N, C // self.num_heads

        v = self.v_lif(v).flatten(3)
        v = (
            v.transpose(-1, -2)
            .reshape(T, B, N, self.num_heads, C // self.num_heads)
            .permute(0, 1, 3, 2, 4)
            .contiguous()
        )

        q = q.mean(dim=-1, keepdim=True)

        k = k.mean(dim=-1, keepdim=True).mean(dim=-2, keepdim=True)
        
        x = (q * k) ## T, B, self.num_heads, N, 1 

        weight = self.fc.weight.to(device=k.device)
        h_seq = torch.addmm(self.fc.bias.unsqueeze(1), weight, x.flatten(1))
        qk = self.surrogate_function(h_seq).view(x.shape)

        x = qk * v

        x = x.transpose(3, 4).reshape(T, B, C, H, W).contiguous()

        x = self.proj_conv(x).reshape(T, B, C, H, W)

        x = x + x_clone
        return x

class MS_MLP(nn.Module):
    def __init__(
        self, in_features, hidden_features=None, out_features=None, drop=0.0, layer=0
    ):
        super().__init__()
        out_features = out_features or in_features
        hidden_features = hidden_features or in_features

        self.fc1_conv = nn.Conv1d(in_features, hidden_features, kernel_size=1, stride=1)
        self.fc1_bn = nn.BatchNorm1d(hidden_features)
        self.fc1_lif = MultiStepLIFNode(tau=2.0, detach_reset=True, backend="cupy")

        self.fc2_conv = nn.Conv1d(
            hidden_features, out_features, kernel_size=1, stride=1
        )
        self.fc2_bn = nn.BatchNorm1d(out_features)

        self.fc2_lif = MultiStepLIFNode(tau=2.0, detach_reset=True, backend="cupy")

        self.c_hidden = hidden_features
        self.c_output = out_features

    def forward(self, x):
        T, B, C, H, W = x.shape
        N = H * W
        x_clone = x.clone()

        x = x.flatten(3)
        x = self.fc1_lif(x)
        x = self.fc1_conv(x.flatten(0, 1))
        x = self.fc1_bn(x).reshape(T, B, self.c_hidden, N).contiguous()

        x = self.fc2_lif(x)
        x = self.fc2_conv(x.flatten(0, 1))
        x = self.fc2_bn(x).reshape(T, B, C, H, W).contiguous()
        x = x + x_clone
        return x

class MS_Block(nn.Module):
    def __init__(
        self,
        dim,
        num_heads=8,
        mlp_ratio=4.0):
        super().__init__()

        self.attn = MS_Attention(dim, num_heads=num_heads)
        mlp_hidden_dim = int(dim * mlp_ratio)
        self.mlp = MS_MLP(in_features=dim, hidden_features=mlp_hidden_dim)

    def forward(self, x):

        x = self.attn(x)
        
        x = self.mlp(x)

        return x

class Spikformer(nn.Module):
    def __init__(self,
                 img_size_h=32, img_size_w=32, patch_size=4, in_channels=3, num_classes=10,
                 embed_dims=[64, 128, 256], num_heads=8, mlp_ratios=4, qkv_bias=False, qk_scale=None,
                 drop_rate=0., attn_drop_rate=0., drop_path_rate=0., norm_layer=nn.LayerNorm,
                 depths=2, sr_ratios=[8, 4, 2], T = 4, kd=False
                 ):
        super().__init__()
        self.num_classes = num_classes
        self.depths = depths
        self.T = 4
        # embed_dim = [64, 128, 256, 512]

        dpr = [
            x.item() for x in torch.linspace(0, drop_path_rate, 2)
        ]  # stochastic depth decay rule

        self.downsample1 = DownSampling(
            in_channels,
            embed_dims[0],
            first_layer=True,
        )

        self.ConvBlock1 = nn.ModuleList(
            [MS_Block(dim=embed_dims[0])]
        )

        self.downsample2 = DownSampling(
            embed_dims[0],
            embed_dims[1],
            first_layer=False,
        )

        self.ConvBlock2 = nn.ModuleList(
            [MS_Block(dim=embed_dims[1])]
        )

        self.downsample3 = DownSampling(
            embed_dims[1],
            embed_dims[2],
            first_layer=False,
        )

        self.block3 = nn.ModuleList(
            [
                MS_Block(dim=embed_dims[2])
                for j in range(2)
            ]
        )

        self.lif = MultiStepLIFNode(tau=2.0, detach_reset=True, backend="cupy")
        self.head = (
            nn.Linear(embed_dims[2], num_classes) if num_classes > 0 else nn.Identity()
        )

        self.kd = kd
        if self.kd:
            self.head_kd = (
                nn.Linear(embed_dims[2], num_classes)
                if num_classes > 0
                else nn.Identity()
            )
        self.apply(self._init_weights)

    def _init_weights(self, m):
        if isinstance(m, nn.Linear):
            trunc_normal_(m.weight, std=0.02)
            if isinstance(m, nn.Linear) and m.bias is not None:
                nn.init.constant_(m.bias, 0)
        elif isinstance(m, nn.LayerNorm):
            nn.init.constant_(m.bias, 0)
            nn.init.constant_(m.weight, 1.0)

    def forward_features(self, x):

        x = self.downsample1(x)
        for blk in self.ConvBlock1:
            x = blk(x)

        x = self.downsample2(x)
        for blk in self.ConvBlock2:
            x = blk(x)

        x = self.downsample3(x)
        for blk in self.block3:
            x = blk(x)
            
        return x  # T,B,C,N

    def forward(self, x):
        x = (x.unsqueeze(0)).repeat(self.T, 1, 1, 1, 1)
        x = self.forward_features(x)
        x = x.flatten(3).mean(3)
        x = self.head(x.mean(dim=0))
        return x

@register_model
def spikformer(pretrained=False, **kwargs):
    model = Spikformer(
        **kwargs
    )
    model.default_cfg = _cfg()
    return model


if __name__=='__main__':
    model = create_model('spikformer', pretrained=False).cuda()
    x = torch.randn((16, 3, 32, 32)).cuda()
    print(model(x).shape)


