import argparse
import sys
import matplotlib.pyplot as plt
from matplotlib.widgets import Slider, Button, RadioButtons
import numpy as np
import torch as th
import torch.nn as nn
import cv2
import pickle
from utils.utils import Gaus2D
from einops import rearrange, repeat, reduce
from utils.configuration import Configuration
from model.lightning.pretrainer import LociPretrainerModule

class LociPlayground:
    
    def __init__(self, cfg, device, file, gestalt = None, position = None):

        device = th.device(device)

        self.cfg      = cfg
        self.gestalt  = th.zeros((1, cfg.model.encoder.gestalt_size)).to(device)
        self.position = th.tensor([[0,0,0,0.05]]).to(device)
        self.size     = cfg.model.input_size
        self.gaus2d   = Gaus2D(cfg.model.input_size).to(device)
        self.gestalt_gridcell_size   = 25
        self.gestalt_gridcell_margin = 5
        self.gestalt_grid_width      = 32
        self.gestalt_grid_height     = 8

        if gestalt is not None:
            self.gestalt = th.from_numpy(gestalt).to(device)

        if position is not None:
            self.position = th.from_numpy(position).to(device)

        model = LociPretrainerModule.load_from_checkpoint(file, cfg=cfg).to(device)
        self.net = model.net

        self.fig = plt.figure(figsize=(6,6))

        self.ax_gestalt  = plt.subplot2grid((3, 3), (0, 0), colspan=2)
        self.ax_position = plt.subplot2grid((3, 3), (0, 2))
        self.ax_output1  = plt.subplot2grid((3, 3), (1, 0), colspan=2, rowspan=2)
        self.ax_output2  = plt.subplot2grid((3, 3), (1, 2))
        self.ax_output3  = plt.subplot2grid((3, 3), (2, 2))

        self.outputs = [self.ax_output1, self.ax_output2, self.ax_output3]
        self.indices = [0, 1, 2]

        self.connections = ()

        self.add_image(self.ax_gestalt, self.create_gestalt_image())
        self.add_image(self.ax_position, self.create_position_image())

        self.update_outputs()

        plt.tight_layout()

    def update_outputs(self):
        mask, object, depth = None, None, None
        with th.no_grad():
            mask, object, depth = self.net.run_decoder(self.position, self.gestalt)
            #object = mask * object
            #depth  = th.zeros_like(depth)

        object = th.sigmoid(object - 2.5)

        self.add_image(self.outputs[self.indices[0]], depth.cpu().numpy()[0].transpose(1,2,0))
        self.add_image(self.outputs[self.indices[1]], object.cpu().numpy()[0].transpose(1,2,0))
        self.add_image(self.outputs[self.indices[2]], mask.cpu().numpy()[0].transpose(1,2,0))

    def __enter__(self):
        self.connections = (
            self.fig.canvas.mpl_connect('button_press_event', self.onclick),
            self.fig.canvas.mpl_connect('scroll_event', self.onscroll)
        )
        return self

    def __exit__(self, *args, **kwargs):
        for connection in self.connections:
            self.fig.canvas.mpl_disconnect(connection)

    def create_gestalt_image(self):

        gestalt = self.gestalt[0].cpu().numpy()
        size    = self.gestalt_gridcell_size
        margin  = self.gestalt_gridcell_margin
            
        width  = self.gestalt_grid_width  * (margin + size) + margin
        height = self.gestalt_grid_height * (margin + size) + margin
        img = np.zeros((height, width, 3)) + 0.3
        
        for i in range(gestalt.shape[0]):
            h = i // self.gestalt_grid_width
            w = i  % self.gestalt_grid_width
            
            img[h*size+(h+1)*margin:(h+1)*(margin+size),w*size+(w+1)*margin:(w+1)*(margin+size),0] = (1 - gestalt[i]) * 0.8 + gestalt[i] * 0.2
            img[h*size+(h+1)*margin:(h+1)*(margin+size),w*size+(w+1)*margin:(w+1)*(margin+size),1] = gestalt[i] * 0.8 + (1 - gestalt[i]) * 0.2
            img[h*size+(h+1)*margin:(h+1)*(margin+size),w*size+(w+1)*margin:(w+1)*(margin+size),2] = 0.2


        return img

    def create_position_image(self):
        
        img = self.gaus2d(self.position)
        img = rearrange(img[0], 'c h w -> h w c')

        return th.cat((img, img, img * 0.6 + 0.4), dim=2).cpu().numpy()


    def add_image(self, ax, img):
        ax.clear()
        ax.imshow(img)
        ax.axis('off')



    def onclick(self, event):
        x, y = event.xdata, event.ydata

        if self.ax_gestalt == event.inaxes:

            size    = self.gestalt_gridcell_size
            margin  = self.gestalt_gridcell_margin

            w = int(x / (margin + size))
            h = int(y / (margin + size))

            i = h * self.gestalt_grid_width + w
            self.gestalt[0,i] = 1 - self.gestalt[0,i]

            self.add_image(self.ax_gestalt, self.create_gestalt_image())
            self.update_outputs()
            self.fig.canvas.draw()

        if self.ax_position == event.inaxes:

            x = (x / self.size[1]) * 2 - 1
            y = (y / self.size[0]) * 2 - 1

            self.position[0,0] = x
            self.position[0,1] = y

            self.add_image(self.ax_position, self.create_position_image())
            self.update_outputs()
            self.fig.canvas.draw()

        if self.ax_output2 == event.inaxes:
            ax_tmp = self.indices[0]
            self.indices[0] = self.indices[1]
            self.indices[1] = ax_tmp
            self.update_outputs()
            self.fig.canvas.draw()

        if self.ax_output3 == event.inaxes:
            ax_tmp = self.indices[0]
            self.indices[0] = self.indices[2]
            self.indices[2] = ax_tmp
            self.update_outputs()
            self.fig.canvas.draw()
            
    def onscroll(self, event):
        if self.ax_position == event.inaxes:
            std = self.position[0,3]
            if event.button == 'down':
                self.position[0,3] = std - 0.1
                
            elif event.button == 'up':
                self.position[0,3] = std + 0.1

            self.add_image(self.ax_position, self.create_position_image())
            self.update_outputs()
            self.fig.canvas.draw()

if __name__=="__main__":

    parser = argparse.ArgumentParser()
    parser.add_argument("-cfg", "--cfg", required=True, type=str)
    parser.add_argument("-load", "--load", required=True, type=str)
    parser.add_argument("-latent", "--latent", default="", type=str)
    parser.add_argument("-device", "--device", default=0, type=int)

    args = parser.parse_args(sys.argv[1:])
    cfg  = Configuration(args.cfg)

    gestalt  = None
    position = None
    if args.latent != "":
        with open(args.latent, 'rb') as infile:
            state = pickle.load(infile)
            gestalt  = state["gestalt"]
            position = state["position"]

    with LociPlayground(cfg, args.device, args.load, gestalt, position):
        plt.show()
