#!/usr/bin/env python3
"""
Example usage script for the Search Evaluation Toolkit
"""

import os
import sys
import json
import subprocess
from pathlib import Path

def run_evaluation_example():
    """Example of running evaluation"""
    print("=== Running Evaluation Example ===")
    
    # Example command for evaluation
    cmd = [
        "python", "evaluation/simple_eval.py",
        "--model_path", "Qwen/Qwen2.5-7B-Instruct",
        "--data_path", "data/example/process_test.parquet",
        "--output_path", "eval_results/example/qwen_results.jsonl",
        "--do_search",
        "--val_batch_size", "16"
    ]
    
    print("Example evaluation command:")
    print(" ".join(cmd))
    print("\nNote: This is just an example. Make sure you have:")
    print("1. The model available")
    print("2. The data file in the correct location")
    print("3. Sufficient GPU memory")

def run_annotation_example():
    """Example of running annotation"""
    print("\n=== Running Annotation Example ===")
    
    # Example command for annotation
    cmd = [
        "python", "annotation/main.py",
        "--input_file", "eval_results/example/qwen_results.jsonl",
        "--output_file", "annotated_results/example/qwen_annotated.jsonl",
        "--ontology_file", "annotation/ontology.json",
        "--model", "gpt-4o-mini",
        "--concurrency", "5"
    ]
    
    print("Example annotation command:")
    print(" ".join(cmd))
    print("\nNote: Make sure you have:")
    print("1. OpenAI API key set in .env file")
    print("2. Input file exists from evaluation step")

def run_analysis_example():
    """Example of running analysis"""
    print("\n=== Running Analysis Example ===")
    
    # Example commands for analysis
    analysis_commands = [
        {
            "name": "RQI and SEI Analysis (O1)",
            "cmd": [
                "python", "analysis/O1_RL_training_impact_on_reasoning_search.py",
                "--input", "annotated_results/example/qwen_annotated.jsonl",
                "--outdir", "analysis_results/figs_cohort",
                "--require_clear_for_sufficient"
            ]
        },
        {
            "name": "Recovery Dynamics Analysis (O2)",
            "cmd": [
                "python", "analysis/O2_Recovery_dynamics.py",
                "annotated_results/example/qwen_annotated.jsonl",
                "analysis_results/recovery_figs"
            ]
        },
        {
            "name": "Evidence Strength Analysis (O3)",
            "cmd": [
                "python", "analysis/O3_Evidence_strength_per_turn.py",
                "--input", "annotated_results/example/qwen_annotated.jsonl",
                "--outdir", "analysis_results/evidence_figs"
            ]
        }
    ]
    
    for analysis in analysis_commands:
        print(f"\n{analysis['name']}:")
        print(" ".join(analysis['cmd']))

def create_sample_data():
    """Create sample data structure"""
    print("\n=== Creating Sample Data Structure ===")
    
    # Create sample data directory
    sample_dir = Path("data/example")
    sample_dir.mkdir(parents=True, exist_ok=True)
    
    # Create sample parquet data (simplified)
    sample_data = {
        "question": ["What is the capital of France?"],
        "answer": ["Paris"],
        "context": ["France is a country in Europe. Its capital is Paris."]
    }
    
    print(f"Created sample data directory: {sample_dir}")
    print("Note: You'll need to create actual parquet files with your data")
    print("The data should have columns: question, answer, context (at minimum)")

def main():
    """Main example function"""
    print("Search Evaluation Toolkit - Example Usage")
    print("=" * 50)
    
    # Show examples
    run_evaluation_example()
    run_annotation_example()
    run_analysis_example()
    create_sample_data()
    
    print("\n" + "=" * 50)
    print("Complete Pipeline Example:")
    print("1. Run evaluation: python evaluation/simple_eval.py [options]")
    print("2. Run annotation: python annotation/main.py [options]")
    print("3. Run analysis: python analysis/O1_RL_training_impact_on_reasoning_search.py [options]")
    print("\nFor detailed usage, see README.md")

if __name__ == "__main__":
    main()
