import diffuser.utils as utils
from diffuser.utils.trajectory import Trajectory
from diffuser.utils.trajectory import ReplayBuffer
import torch
from ml_logger import logger, RUN
from config.locomotion_config import Config
from copy import deepcopy
import numpy as np
import pickle
import os
import gym
from diffuser.models.value_func_model import ValueMLP
from diffuser.utils.arrays import to_torch, to_np, to_device
from diffuser.models.helpers import BellmanLoss
import random
from scripts.create_trajectory import train_val


def save_traj(traj, bucket):
    savepath = os.path.join(bucket, logger.prefix, 'checkpoint')
    os.makedirs(savepath, exist_ok=True)
    savepath = os.path.join(savepath, 'trajectories.dat')
    with open(savepath, "wb") as f:
        pickle.dump(traj, f)


def save(model1, model2, bucket):
    data = {
        'model1': model1.state_dict(),
        'model2': model2.state_dict()
    }
    savepath = os.path.join(bucket, logger.prefix, 'checkpoint')
    os.makedirs(savepath, exist_ok=True)
    # logger.save_torch(data, savepath)

    savepath = os.path.join(savepath, 'V_model.pt')
    torch.save(data, savepath)


def set_seed(seed, deterministic_torch=False):
    os.environ["PYTHONHASHSEED"] = str(seed)
    np.random.seed(seed)
    random.seed(seed)
    torch.manual_seed(seed)
    torch.use_deterministic_algorithms(deterministic_torch)


def train(**deps):

    RUN._update(deps)
    Config._update(deps)

    logger.remove('*.pkl')
    logger.remove("traceback.err")
    logger.log_params(Config=vars(Config), RUN=vars(RUN))

    Config.device = 'cuda'

    if Config.predict_epsilon:
        prefix = f'predict_epsilon_{Config.n_diffusion_steps}_1000000.0'
    else:
        prefix = f'predict_x0_{Config.n_diffusion_steps}_1000000.0'

    loadpath = os.path.join(Config.bucket, logger.prefix, 'checkpoint')

    if Config.save_checkpoints:
        loadpath = os.path.join(loadpath, f'state_{self.step}.pt')
    else:
        loadpath = os.path.join(loadpath, 'state.pt')

    state_dict = torch.load(loadpath, map_location=Config.device)

    # Load configs
    torch.backends.cudnn.benchmark = True
    utils.set_seed(Config.seed)

    dataset_config = utils.Config(
        Config.loader,
        savepath='dataset_config.pkl',
        env=Config.dataset,
        horizon=Config.horizon,
        normalizer=Config.normalizer,
        preprocess_fns=Config.preprocess_fns,
        use_padding=Config.use_padding,
        max_path_length=Config.max_path_length,
        include_returns=Config.include_returns,
        returns_scale=Config.returns_scale,
    )

    render_config = utils.Config(
        Config.renderer,
        savepath='render_config.pkl',
        env=Config.dataset,
    )

    dataset = dataset_config()
    renderer = render_config()

    observation_dim = dataset.observation_dim
    action_dim = dataset.action_dim

    if Config.diffusion == 'models.GaussianInvDynDiffusion':
        transition_dim = observation_dim
    else:
        transition_dim = observation_dim + action_dim

    model_config = utils.Config(
        Config.model,
        savepath='model_config.pkl',
        horizon=Config.horizon,
        transition_dim=transition_dim,
        cond_dim=observation_dim,
        dim_mults=Config.dim_mults,
        dim=Config.dim,
        returns_condition=Config.returns_condition,
        device=Config.device,
    )

    diffusion_config = utils.Config(
        Config.diffusion,
        savepath='diffusion_config.pkl',
        horizon=Config.horizon,
        observation_dim=observation_dim,
        action_dim=action_dim,
        n_timesteps=Config.n_diffusion_steps,
        loss_type=Config.loss_type,
        clip_denoised=Config.clip_denoised,
        predict_epsilon=Config.predict_epsilon,
        hidden_dim=Config.hidden_dim,
        ## loss weighting
        action_weight=Config.action_weight,
        loss_weights=Config.loss_weights,
        loss_discount=Config.loss_discount,
        returns_condition=Config.returns_condition,
        device=Config.device,
        condition_guidance_w=Config.condition_guidance_w,
    )

    trainer_config = utils.Config(
        utils.Trainer,
        savepath='trainer_config.pkl',
        train_batch_size=Config.batch_size,
        train_lr=Config.learning_rate,
        gradient_accumulate_every=Config.gradient_accumulate_every,
        ema_decay=Config.ema_decay,
        sample_freq=Config.sample_freq,
        save_freq=Config.save_freq,
        log_freq=Config.log_freq,
        label_freq=int(Config.n_train_steps // Config.n_saves),
        save_parallel=Config.save_parallel,
        bucket=Config.bucket,
        n_reference=Config.n_reference,
        train_device=Config.device,
    )

    model = model_config()
    diffusion = diffusion_config(model)
    trainer = trainer_config(diffusion, dataset, renderer)
    logger.print(utils.report_parameters(model), color='green')
    trainer.step = state_dict['step']
    trainer.model.load_state_dict(state_dict['model'])
    trainer.ema_model.load_state_dict(state_dict['ema'])

    num_eval = 100
    device = Config.device

    env_list = [gym.make(Config.dataset) for _ in range(num_eval)]
    dones = [0 for _ in range(num_eval)]
    episode_rewards = [0 for _ in range(num_eval)]

    assert trainer.ema_model.condition_guidance_w == Config.condition_guidance_w
    returns = to_device(Config.test_ret * torch.ones(num_eval, 1), device)

    t = 0
    obs_list = [env.reset()[None] for env in env_list]
    obs = np.concatenate(obs_list, axis=0)
    recorded_obs = [deepcopy(obs[:, None])]

    reward_list = [[] for i in range(num_eval)]
    true_list = [[] for i in range(num_eval)]

    traj_length = 2000

    path_num = dataset.fields.normed_observations.shape[0]
    state_dim = env_list[0].observation_space.shape[0]
    action_dim = env_list[0].action_space.shape[0]
    length = [0 for i in range(num_eval)]
    traj = [Trajectory(length=traj_length, state_dim=state_dim, action_dim=action_dim) for i in range(path_num)]

    # loadpath = os.path.join(Config.bucket, logger.prefix, 'checkpoint')
    # os.makedirs(loadpath, exist_ok=True)
    # loadpath = os.path.join(loadpath, 'trajectories.dat')
    # with open(loadpath, "rb") as f:
    #     traj = pickle.load(f)


    paths = []

    for path_ind in range(path_num):
        # path_ind, start, end = dataset.indices[i]
        observations = dataset.fields.normed_observations[path_ind, :dataset.fields.path_lengths[path_ind]]
        if path_ind == 0:
            obs_numpy = observations
        else:
            obs_numpy = np.concatenate((obs_numpy, observations), axis=0)
        actions = dataset.fields.normed_actions[path_ind, :dataset.fields.path_lengths[path_ind]]
        rewards = dataset.fields.rewards[path_ind] / 4
        next_item = np.concatenate((dataset.fields.normed_observations[path_ind, 1:dataset.fields.path_lengths[path_ind]],
                                    np.array([dataset.fields.normed_observations[path_ind, dataset.fields.path_lengths[path_ind] - 1]])), axis=0)
        paths.append([observations, actions, rewards, next_item])

    for path_ind in range(path_num):
        length = paths[path_ind][0].shape[0]
        for i in range(length):
            traj[path_ind].add_item(input_state=paths[path_ind][0][i], input_action=paths[path_ind][1][i], input_reward=paths[path_ind][2][i], idx=i)


    train_val(traj, state_dim, action_dim, device, Config.seed, Config.bucket)