import argparse
import os
import random
import shutil
import time
import warnings

import torch
import torch.nn as nn
import torch.nn.parallel
import torch.backends.cudnn as cudnn
import torch.distributed as dist
import torch.optim
import torch.multiprocessing as mp
import torch.utils.data
import torch.utils.data.distributed
import torchvision.transforms as transforms
import torchvision.datasets as datasets
import torchvision.models as models
from torch.utils.tensorboard import SummaryWriter
from model import resnet,vgg
import numpy as np
import torch.nn.functional as F
import matplotlib.pyplot as plt
from utils.visualization import save_images,save_images2

STATUS='Blocking'#'Poison'#
DATASET='test'#'train'#
PATTERN='square'#'cross'#
PARTIAL=False
ROOT='result'
if not os.path.exists(ROOT):
    os.mkdir(ROOT)


SIZE=1

torch.set_num_threads(4)
os.environ["CUDA_VISIBLE_DEVICES"] = "7"

def setup_seed(seed):
    torch.manual_seed(seed)
    torch.cuda.manual_seed_all(seed)
    np.random.seed(seed)
    random.seed(seed)

model_names = sorted(name for name in models.__dict__
    if name.islower() and not name.startswith("__")
    and callable(models.__dict__[name]))

parser = argparse.ArgumentParser(description='PyTorch ImageNet Training')
parser.add_argument('--data', metavar='DIR',default='/data_SSD2/zgh/workspace/data/ImageNet',
                    help='path to dataset')
parser.add_argument('-a', '--arch', metavar='ARCH', default='resnet50',
                    choices=model_names,
                    help='model architecture: ' +
                        ' | '.join(model_names) +
                        ' (default: resnet18)')
parser.add_argument('-j', '--workers', default=4, type=int, metavar='N',
                    help='number of data loading workers (default: 4)')
parser.add_argument('--epochs', default=90, type=int, metavar='N',
                    help='number of total epochs to run')
parser.add_argument('--start-epoch', default=0, type=int, metavar='N',
                    help='manual epoch number (useful on restarts)')
parser.add_argument('-b', '--batch-size', default=16, type=int,
                    metavar='N',
                    help='mini-batch size (default: 256), this is the total '
                         'batch size of all GPUs on the current node when '
                         'using Data Parallel or Distributed Data Parallel')
parser.add_argument('--lr', '--learning-rate', default=0.1, type=float,
                    metavar='LR', help='initial learning rate', dest='lr')
parser.add_argument('--momentum', default=0.9, type=float, metavar='M',
                    help='momentum')
parser.add_argument('--wd', '--weight-decay', default=1e-4, type=float,
                    metavar='W', help='weight decay (default: 1e-4)',
                    dest='weight_decay')
parser.add_argument('-p', '--print-freq', default=10, type=int,
                    metavar='N', help='print frequency (default: 10)')
parser.add_argument('--resume', default='', type=str, metavar='PATH',
                    help='path to latest checkpoint (default: none)')
parser.add_argument('-e', '--evaluate', dest='evaluate', action='store_true',
                    help='evaluate model on validation set')
parser.add_argument('--pretrained', dest='pretrained', action='store_true',
                    help='use pre-trained model')
parser.add_argument('--world-size', default=-1, type=int,
                    help='number of nodes for distributed training')
parser.add_argument('--rank', default=-1, type=int,
                    help='node rank for distributed training')
parser.add_argument('--dist-url', default='tcp://224.66.41.62:23456', type=str,
                    help='url used to set up distributed training')
parser.add_argument('--dist-backend', default='nccl', type=str,
                    help='distributed backend')
parser.add_argument('--seed', default=None, type=int,
                    help='seed for initializing training. ')
parser.add_argument('--gpu', default=None, type=int,
                    help='GPU id to use.')
parser.add_argument('--multiprocessing-distributed', action='store_true',
                    help='Use multi-processing distributed training to launch '
                         'N processes per node, which has N GPUs. This is the '
                         'fastest way to use PyTorch for either single node or '
                         'multi node data parallel training')

def main():
    args = parser.parse_args()

    if args.seed is not None:
        random.seed(args.seed)
        torch.manual_seed(args.seed)
        cudnn.deterministic = True
        warnings.warn('You have chosen to seed training. '
                      'This will turn on the CUDNN deterministic setting, '
                      'which can slow down your training considerably! '
                      'You may see unexpected behavior when restarting '
                      'from checkpoints.')

    if args.gpu is not None:
        warnings.warn('You have chosen a specific GPU. This will completely '
                      'disable data parallelism.')

    if args.dist_url == "env://" and args.world_size == -1:
        args.world_size = int(os.environ["WORLD_SIZE"])

    args.distributed = args.world_size > 1 or args.multiprocessing_distributed

    ngpus_per_node = torch.cuda.device_count()

    cudnn.benchmark = True

    traindir = os.path.join(args.data, 'train')
    valdir = os.path.join(args.data, 'val')
    normalize = transforms.Normalize(mean=[0.485, 0.456, 0.406],
                                     std=[0.229, 0.224, 0.225])
    '''
    train_dataset = datasets.ImageFolder(
        traindir,
        transforms.Compose([
            transforms.Resize(256),
            transforms.CenterCrop(224),
            transforms.ToTensor(),
            normalize,
        ]))
    if args.distributed:
        train_sampler = torch.utils.data.distributed.DistributedSampler(train_dataset)
    else:
        train_sampler = None
    train_loader = torch.utils.data.DataLoader(
        train_dataset, batch_size=args.batch_size, shuffle=(train_sampler is None),
        num_workers=args.workers, pin_memory=True, sampler=train_sampler)
    '''
    setup_seed(999)
    val_loader = torch.utils.data.DataLoader(
        datasets.ImageFolder(valdir, transforms.Compose([
            transforms.Resize(256),
            transforms.CenterCrop(224),
            transforms.ToTensor(),
            normalize,
        ])),
        batch_size=args.batch_size, shuffle=True,
        num_workers=args.workers, pin_memory=False)

    main_worker(val_loader,SIZE, args)

class Block_Layer(nn.Module):
    def __init__(self,size=1,position=(0,0)):
        super(Block_Layer,self).__init__()
        self.relu=nn.ReLU()
        self.size=size
        self.position=position

    def forward(self, x):
        mask = torch.ones_like(x)
        s = int(self.size * x.size(2) / 7)
        a,b=self.position
        mask[:, :, a*s:(a+1)*s, b*s:(b+1)*s] = 0
        x = self.relu(x * mask)
        return x

def blocking_model(model,target_layer=2,size=1):
    print(len(model.features))
    index=-1
    for i in range(len(model.features)):
        if isinstance(model.features[i], nn.ReLU):
            index+=1
            if index + target_layer == 12:
                model.features[i]=Block_Layer(size)
    return model

def energy_rate(x,y,size=1,position=(0,0)):
    mask=torch.ones_like(x)
    s=int(size*x.size(2)/7)
    a, b = position
    mask[:,:,a*s:(a+1)*s+1, b*s:(b+1)*s+1]=0
    x=x*mask
    y=y*mask
    er=torch.sum(torch.abs(x-y))/(torch.sum(torch.abs(x)))
    return er.item()

def obtain_activations(x,model):
    activation_maps = []
    handle=[]

    def forward_hook_fn(module, input, output):
        # 在全局变量中保存 ReLU 层的前向传播输出
        # 用于将来做 guided backpropagation
        if len(output.size())==4:
            activation_maps.append(output)

    for m in model.modules():
        if isinstance(m, nn.ReLU) or isinstance(m, nn.LeakyReLU):
            h = m.register_forward_hook(forward_hook_fn)
            handle.append(h)

    with torch.no_grad():
        out=model(x)
    for h in handle:
        h.remove()
    return activation_maps

def main_worker(val_loader,size, args):
    # create model
    model=vgg.vgg16(pretrained=True).cuda()

    models=[]
    import copy
    model.eval()
    for i in range(13):
        m=copy.deepcopy(model)
        models.append(blocking_model(m,i,size))

    if DATASET=='train':
        loader=None
    else:
        loader=val_loader
    pth = os.path.join(ROOT, STATUS,PATTERN+'_'+str(size))
    if not os.path.exists(os.path.join(ROOT, STATUS)):
        os.mkdir(os.path.join(ROOT, STATUS))
    if not os.path.exists(pth):
        os.mkdir(pth)
    if True:
        end = time.time()
        ER=np.zeros([13,13])
        for i, (images, target) in enumerate(loader):
            images, target = images.cuda(), target.cuda()
            base_activation=obtain_activations(images,model)

            for j in range(len(models)):
                temp=obtain_activations(images,models[j])
                for k in range(len(temp)):
                    ER[k,j]=(ER[k,j]*i+energy_rate(base_activation[12-k],temp[12-k],SIZE))/(i+1)

            print(i,ER)
            draw(ER,os.path.join(pth,'ER.jpg'))
            np.save(os.path.join(pth, 'ER.npy'), ER)

def draw(data,pth):
    fig=plt.figure()
    ax=fig.add_subplot(111)
    im=ax.imshow(data)#cmap=plt.cm.hot_r
    plt.colorbar(im)
    for  i in range(data.shape[0]):
        for j in  range(data.shape[1]):
            ax.text(j,i,int(data[i,j]*100)/100.0,ha='center',va='center',color='w')
    fig.tight_layout()
    plt.savefig(pth)
    plt.close()


if __name__ == '__main__':
    main()