import argparse
import os
import random
import shutil
import time
import warnings
import json
import torch
import torch.nn as nn
import torch.nn.parallel
import torch.backends.cudnn as cudnn
import torch.distributed as dist
import torch.optim
import torch.multiprocessing as mp
import torch.utils.data
import torch.utils.data.distributed
import torchvision.transforms as transforms
import torchvision.datasets as datasets
import torchvision.models as models
from torch.utils.tensorboard import SummaryWriter
from model import resnet,vgg
import numpy as np
import torch.nn.functional as F
import matplotlib.pyplot as plt
from utils.visualization import save_images5,save_images2,save_images3


def setup_seed(seed):
    torch.manual_seed(seed)
    torch.cuda.manual_seed_all(seed)
    np.random.seed(seed)
    random.seed(seed)



def main():
    model_names = sorted(name for name in models.__dict__
                         if name.islower() and not name.startswith("__")
                         and callable(models.__dict__[name]))

    parser = argparse.ArgumentParser(description='PyTorch ImageNet Training')
    parser.add_argument('--data', metavar='DIR', default='/data_SSD2/zgh/workspace/data/ImageNet',
                        help='path to dataset')
    parser.add_argument('-a', '--arch', metavar='ARCH', default='resnet50',
                        choices=model_names,
                        help='model architecture: ' +
                             ' | '.join(model_names) +
                             ' (default: resnet18)')
    parser.add_argument('-j', '--workers', default=4, type=int, metavar='N',
                        help='number of data loading workers (default: 4)')
    parser.add_argument('--epochs', default=90, type=int, metavar='N',
                        help='number of total epochs to run')
    parser.add_argument('--start-epoch', default=0, type=int, metavar='N',
                        help='manual epoch number (useful on restarts)')
    parser.add_argument('-b', '--batch-size', default=BATCH, type=int,
                        metavar='N',
                        help='mini-batch size (default: 256), this is the total '
                             'batch size of all GPUs on the current node when '
                             'using Data Parallel or Distributed Data Parallel')
    parser.add_argument('--lr', '--learning-rate', default=0.1, type=float,
                        metavar='LR', help='initial learning rate', dest='lr')
    parser.add_argument('--momentum', default=0.9, type=float, metavar='M',
                        help='momentum')
    parser.add_argument('--wd', '--weight-decay', default=1e-4, type=float,
                        metavar='W', help='weight decay (default: 1e-4)',
                        dest='weight_decay')
    parser.add_argument('-p', '--print-freq', default=10, type=int,
                        metavar='N', help='print frequency (default: 10)')
    parser.add_argument('--resume', default='', type=str, metavar='PATH',
                        help='path to latest checkpoint (default: none)')
    parser.add_argument('-e', '--evaluate', dest='evaluate', action='store_true',
                        help='evaluate model on validation set')
    parser.add_argument('--pretrained', dest='pretrained', action='store_true',
                        help='use pre-trained model')
    parser.add_argument('--world-size', default=-1, type=int,
                        help='number of nodes for distributed training')
    parser.add_argument('--rank', default=-1, type=int,
                        help='node rank for distributed training')
    parser.add_argument('--dist-url', default='tcp://224.66.41.62:23456', type=str,
                        help='url used to set up distributed training')
    parser.add_argument('--dist-backend', default='nccl', type=str,
                        help='distributed backend')
    parser.add_argument('--seed', default=None, type=int,
                        help='seed for initializing training. ')
    parser.add_argument('--gpu', default=None, type=int,
                        help='GPU id to use.')
    parser.add_argument('--multiprocessing-distributed', action='store_true',
                        help='Use multi-processing distributed training to launch '
                             'N processes per node, which has N GPUs. This is the '
                             'fastest way to use PyTorch for either single node or '
                             'multi node data parallel training')

    args = parser.parse_args()

    if args.seed is not None:
        random.seed(args.seed)
        torch.manual_seed(args.seed)
        cudnn.deterministic = True
        warnings.warn('You have chosen to seed training. '
                      'This will turn on the CUDNN deterministic setting, '
                      'which can slow down your training considerably! '
                      'You may see unexpected behavior when restarting '
                      'from checkpoints.')

    if args.gpu is not None:
        warnings.warn('You have chosen a specific GPU. This will completely '
                      'disable data parallelism.')

    if args.dist_url == "env://" and args.world_size == -1:
        args.world_size = int(os.environ["WORLD_SIZE"])

    args.distributed = args.world_size > 1 or args.multiprocessing_distributed

    ngpus_per_node = torch.cuda.device_count()

    cudnn.benchmark = True

    traindir = os.path.join(args.data, 'train')
    valdir = os.path.join(args.data, 'val')
    normalize = transforms.Normalize(mean=[0.485, 0.456, 0.406],
                                     std=[0.229, 0.224, 0.225])
    '''
    train_dataset = datasets.ImageFolder(
        traindir,
        transforms.Compose([
            transforms.Resize(256),
            transforms.CenterCrop(224),
            transforms.ToTensor(),
            normalize,
        ]))
    if args.distributed:
        train_sampler = torch.utils.data.distributed.DistributedSampler(train_dataset)
    else:
        train_sampler = None
    train_loader = torch.utils.data.DataLoader(
        train_dataset, batch_size=args.batch_size, shuffle=(train_sampler is None),
        num_workers=args.workers, pin_memory=True, sampler=train_sampler)
    '''
    setup_seed(999)
    val_loader = torch.utils.data.DataLoader(
        datasets.ImageFolder(valdir, transforms.Compose([
            transforms.Resize(256),
            transforms.CenterCrop(224),
            transforms.ToTensor(),
            normalize,
        ])),
        batch_size=args.batch_size, shuffle=False,
        num_workers=args.workers, pin_memory=False)

    main_worker(val_loader, args)

def obtain_relu_num(net):
    num=1
    for m in net.modules():
        if isinstance(m, nn.ReLU):
            num+=1
    return num
import copy
import gc
def prepare_model(model,mode='vgg16'):
    model.eval()
    model_As = []
    model_Bs = []
    feature_size = []
    k=1
    if mode=='vgg16':
        L = obtain_relu_num(model)
        for i in range(0, int((L - 2-k)/k)):
            print(i)
            A, B, W = split_model(model, i*k,i*k+k)
            model_As.append(A)
            model_Bs.append(B)
            feature_size.append(W)
            gc.collect()
    else:
        for i in range(17):
            print(i)
            A, B, W = split_model_res(model, i,i+1)
            model_As.append(A)
            model_Bs.append(B)
            feature_size.append(W)
            gc.collect()
    return model_As,model_Bs,feature_size
def _set_module(model, submodule_key, module):
    tokens = submodule_key.split('.')
    sub_tokens = tokens[:-1]
    cur_mod = model
    for s in sub_tokens:
        cur_mod = getattr(cur_mod, s)
    setattr(cur_mod, tokens[-1], module)

def split_model2(net,N):
    if N<1:
        print('illegal N:',N)
    model_A=copy.deepcopy(net)
    model_B=copy.deepcopy(net)
    n=0
    width=0
    for name, module in model_A.named_modules():
        if isinstance(module, nn.ReLU):
            n+=1
        if n>=N :
            if isinstance(module, nn.Conv2d) or isinstance(module, nn.Linear) or isinstance(module, nn.ReLU)\
                    or isinstance(module, nn.AdaptiveAvgPool2d) or isinstance(module, nn.MaxPool2d)\
                    or isinstance(module, nn.BatchNorm2d) or isinstance(module, nn.Dropout):
                _set_module(model_A, name, nn.Sequential())

    n=0
    for name, module in model_B.named_modules():
        if isinstance(module, nn.ReLU):
            n+=1
        if n<N :
            if isinstance(module, nn.Conv2d) or isinstance(module, nn.Linear) or isinstance(module, nn.ReLU)\
                    or isinstance(module, nn.AdaptiveAvgPool2d) or isinstance(module, nn.MaxPool2d)\
                    or isinstance(module, nn.BatchNorm2d) or isinstance(module, nn.Dropout):
                _set_module(model_B, name, nn.Sequential())
            if isinstance(module, nn.Conv2d):
                width=module.out_channels
    return model_A,model_B,width
def split_model(net,N,M):
    if N<0:
        print('illegal N:',N)
    model_A=copy.deepcopy(net)
    model_B=copy.deepcopy(net)
    n=0
    width=0
    for name, module in model_A.named_modules():
        if isinstance(module, nn.ReLU):
            n+=1
        if n<N or n>=M :
            if isinstance(module, nn.Conv2d) or isinstance(module, nn.Linear) or isinstance(module, nn.ReLU)\
                    or isinstance(module, nn.AdaptiveAvgPool2d) or isinstance(module, nn.MaxPool2d)\
                    or isinstance(module, nn.BatchNorm2d) or isinstance(module, nn.Dropout):
                _set_module(model_A, name, nn.Sequential())

    n=0
    for name, module in model_B.named_modules():
        if isinstance(module, nn.ReLU):
            n+=1
        if n<M :
            if isinstance(module, nn.Conv2d) or isinstance(module, nn.Linear) or isinstance(module, nn.ReLU)\
                    or isinstance(module, nn.AdaptiveAvgPool2d) or isinstance(module, nn.MaxPool2d)\
                    or isinstance(module, nn.BatchNorm2d) or isinstance(module, nn.Dropout):
                _set_module(model_B, name, nn.Sequential())
            if isinstance(module, nn.Conv2d):
                width=module.out_channels
    return model_A,model_B,width

def split_model_res(net,N,M):
    layer_name=[['conv1','bn1','relu'],['maxpool','layer1.0'],['layer1.1'],['layer1.2'],
                ['layer2.0'],['layer2.1'],['layer2.2'],['layer2.3'],
                ['layer3.0'],['layer3.1'],['layer3.2'],['layer3.3'],['layer3.4'],['layer3.5'],
                ['layer4.0'],['layer4.1'],['layer4.2'],
                ['avgpool','fc']]
    width=[64,256,256,256,512,512,512,512,1024,1024,1024,1024,1024,1024,2048,2048,2048]
    if N<0:
        print('illegal N:',N)
    model_A=copy.deepcopy(net)
    model_B=copy.deepcopy(net)
    for i in range(len(layer_name)):
        if i<N or i>=M:
            for name in layer_name[i]:
                _set_module(model_A, name, nn.Sequential())
        if i<M:
            for name in layer_name[i]:
                _set_module(model_B, name, nn.Sequential())

    return model_A,model_B,width[N]


def explain_middle_layer_and_raw(base_img,model,model_As,model_Bs,feature_size,methods,target=None):
    import ADGT
    adgt = ADGT.ADGT(use_cuda=True, name='ImageNet')
    result = []
    for m in methods:
        result.append([])
    M=base_img
    for n in range(len(model_As) + 1):
        if n == 0:
            for k, m in enumerate(methods):
                grad = adgt.pure_explain(base_img, model, m,target=target)
                temp = grad.sum(1, keepdim=True)
                maxx, _ = temp.view(temp.size(0), -1).max(1)
                temp = temp / maxx.view(-1,1,1,1)
                temp = temp.detach()
                result[k].append(temp)
                # save_images3(grad.detach().cpu().numpy(), os.path.join(img_pth, m+str(j)+'_'+str(n)+ '.png'))
        else:
            # print(model_As[n-1])
            width = feature_size[n - 1]
            M = model_As[n - 1](M)
            H = int(np.sqrt(M.size(1) / width))
            M = M.view(M.size(0), width, H, H)
            for k, m in enumerate(methods):
                grad = adgt.pure_explain(M, model_Bs[n - 1], m,target=target)
                temp = grad.sum(1, keepdim=True)
                '''
                if temp.size(2) < 224:
                    # temp=F.interpolate(temp,(224,224),mode='nearest')
                    temp = F.interpolate(temp, (224, 224), mode='bilinear')
                maxx, _ = temp.view(temp.size(0), -1).abs().max(1)
                temp = temp / maxx.view(-1,1,1,1)
                '''
                temp = temp.detach()
                result[k].append(temp)
    return result

def cut_most(heatmaps,p=1):
    p=100-p
    heatmaps=heatmaps.float()
    rawgrad = heatmaps.view(heatmaps.size(0), -1).cpu().numpy()
    temp = np.percentile(rawgrad, p, axis=1)

    if len(list(heatmaps.size())) == 4:
        temp = torch.FloatTensor(temp).cuda().view(heatmaps.size(0), 1, 1, 1)
        # temp2 = torch.Tensor(temp2).cuda().view(grad_out[0].size(0), 1, 1, 1)
    elif len(list(heatmaps.size())) == 3:
        temp = torch.FloatTensor(temp).cuda().view(heatmaps.size(0), 1,1)
        # temp2 = torch.Tensor(temp2).cuda().view(grad_out[0].size(0), 1)

    return torch.where(torch.le(heatmaps, temp), heatmaps, temp)/temp

def cut_least0(heatmaps,p=0.0):
    temp=heatmaps
    maxx, _ = torch.max(temp.view(temp.size(0), -1), 1)
    maxx = maxx.view(-1, 1, 1, 1)
    temp = temp / (maxx + 1e-8)
    temp=(torch.clamp_min(temp,p)-p)/(1-p)

    return temp

def cut_least(heatmaps,p=50):
    #p=100-p
    heatmaps=heatmaps.float()
    rawgrad = heatmaps.view(heatmaps.size(0), -1).cpu().numpy()
    temp = np.percentile(rawgrad, p, axis=1)

    if len(list(heatmaps.size())) == 4:
        temp = torch.FloatTensor(temp).cuda().view(heatmaps.size(0), 1, 1, 1)
        # temp2 = torch.Tensor(temp2).cuda().view(grad_out[0].size(0), 1, 1, 1)
    elif len(list(heatmaps.size())) == 3:
        temp = torch.FloatTensor(temp).cuda().view(heatmaps.size(0), 1,1)
        # temp2 = torch.Tensor(temp2).cuda().view(grad_out[0].size(0), 1)

    return torch.where(torch.ge(heatmaps, temp), heatmaps, temp)

def cut_most_least(heatmaps, most_rate,least_rate):
    heatmaps = heatmaps.float()
    default = torch.sign(heatmaps)
    rawgrad = heatmaps.view(heatmaps.size(0), -1).cpu().numpy()
    most_temp,least_temp = np.percentile(rawgrad, (most_rate,least_rate), axis=1)
    #print(most_rate,least_rate,most_temp,least_temp)
    most_temp = torch.FloatTensor(most_temp).cuda().view(-1, 1, 1, 1)
    least_temp = torch.FloatTensor(least_temp).cuda().view(-1, 1, 1, 1)

    heatmaps=torch.where(torch.le(heatmaps, most_temp), heatmaps, most_temp)
    heatmaps=torch.where(torch.ge(heatmaps, least_temp), heatmaps, least_temp)
    heatmaps=(heatmaps-least_temp)/(most_temp-least_temp+1e-8)
    mask=torch.sign(least_temp)
    heatmaps=heatmaps*mask+default*(1-mask)
    return heatmaps

def combine0(X):
    if not isinstance(X,list):
        print('combine target need list')
    N=len(X)
    y=None
    for i,x in enumerate(X):
        temp=x.sum(1,keepdim=True)
        #temp=cut_most(temp,5)
        temp = cut_least(temp, 50)
        if temp.size(2) < 224:
            # temp=F.interpolate(temp,(224,224),mode='nearest')
            temp = F.interpolate(temp, (224, 224), mode='bilinear')
        #temp=F.relu(temp)
        maxx,_=torch.max(temp.view(temp.size(0),-1),1)
        maxx=maxx.view(-1,1,1,1)
        temp=temp/(maxx+1e-8)
        upper=(1-i/N)
        lower=(1-(i+1)/N)
        mask = (temp != 0).float()
        temp = (upper - lower) * temp + lower * mask
        if y is None:
            y = temp
        else:
            # mask=(temp==upper).float()
            yy = y * mask
            y = yy + temp * (yy == 0).float()
    return y

def group(X):
    if not isinstance(X, list):
        print('combine target need list')
    if len(X)<17:
        start = (0, 1, 3, 5, 8, 11, 16)
    else:
        start=(0,1,2,5,9,15,18)
    y_split = []
    index = -1
    for i, x in enumerate(X):
        temp = x.sum(1, keepdim=True)
        '''
        #temp=cut_least(temp,60)

        #temp=cut_most(temp)
        maxx, _ = torch.max(temp.view(temp.size(0), -1), 1)
        maxx = maxx.view(-1, 1, 1, 1)
        temp = temp / (maxx + 1e-8)
        '''
        # print(i,index)
        if i == start[index + 1]:
            y_split.append(temp)
            index = index + 1
        else:
            y_split[index] += temp
    for i in range(len(y_split)):
        temp=F.relu(y_split[i])
        if temp.size(2) < 224:
            # temp=F.interpolate(temp,(224,224),mode='nearest')
            temp = F.interpolate(temp, (224, 224), mode='bilinear')
        y_split[i]=temp

    return y_split
def least_part(y_split):
    result = 0
    for i, y in enumerate(y_split):
        temp =y
        temp = normalize(temp)
        if True:
            result = (result * temp * i + temp) / (i + 1)
    return result

def most_part(y_split,atten=1,alpha=0.5,mode='raw'):
    result_x = 0
    N = len(y_split) * 1.0
    remove_alpha=[]
    threshold_split=[]
    for i, y in enumerate(y_split):
        temp = (y+ 1e-6) * (atten + 1e-6)

        upper = (1 - i / N)
        lower = (1 - (i + 1) / N)

        temp = normalize(temp)
        if mode=='all':
            remove_alpha.append(cut_least(temp,p=int(alpha*100)))
        # temp = temp * (temp > lower).float()
        # temp = torch.clamp_max(temp, upper)
        temp = cut_most_least(temp, (1 - i / N * alpha) * 100.0, (1 - (i + 1) / N * alpha) * 100.0)
        temp = temp * (upper - lower) + lower * (temp != 0)
        if mode=='all':
            threshold_split.append(temp)

        if i == 0:
            result_x = temp
        else:
            result_x = result_x + temp * (result_x == 0).float()
    if mode=='all':
        return result_x,remove_alpha,threshold_split
    else:
        return result_x

def most_part2(y_split,alpha=0,mode='raw'):
    result_x = 0
    N = len(y_split) * 1.0

    remove_alpha=[]
    threshold_split=[]
    for i, y in enumerate(y_split):
        temp=F.relu(y)
        #temp = F.relu(y_split[i])
        if temp.size(2) < 224:
            # temp=F.interpolate(temp,(224,224),mode='nearest')
            temp = F.interpolate(temp, (224, 224), mode='bilinear')

        upper = (1 - i / N)*(1-alpha)+alpha
        lower = (1 - (i + 1) / N)*(1-alpha)+alpha

        #temp = normalize(temp)
        #if mode=='all':
        #    remove_alpha.append(cut_least(temp,p=int(alpha*100)))
        # temp = temp * (temp > lower).float()
        # temp = torch.clamp_max(temp, upper)
        temp = cut_most_least(temp, upper * 100.0, lower * 100.0)
        temp = temp * (upper - lower) #+ lower * (temp != 0)
        threshold_split.append(temp)

        result_x+=temp
    return result_x,threshold_split

def combine(X,alpha=0.5,mode='combine'):

    #y_split=X
    if mode=='least':
        y_split = group(X)
        result=least_part(y_split)
    elif mode=='most':
        result,ts=most_part2(X,alpha=alpha)
        result=(result,ts)
    elif mode=='combine':
        y_split = group(X)
        result=least_part(y_split)
        result = most_part(y_split, result,alpha=alpha)
    elif mode=='all':
        y_split = group(X)
        result0=least_part(y_split)
        result1,ra,ts=most_part(y_split,alpha=alpha,mode='all')
        #result2=most_part(y_split, result0,alpha=alpha)
        #result2=cut_most(result0,5)*result1
        result2=result0*result1
        result=(result0,result1,result2,y_split,ra,ts,)
    else:
        print('mode error: only least, most, combine, all!')
        result=None
    return result

def normalize(temp):
    maxx, _ = torch.max(temp.view(temp.size(0), -1), 1)
    maxx = maxx.view(-1, 1, 1, 1)
    temp = temp / (maxx + 1e-8)
    return temp
def cosine_similarity(x,y):
    x=x.view(x.size(0),-1)
    y=y.view(x.size(0),-1)
    xy=(x*y).sum(1)
    x_norm=(x*x).sum(1).sqrt()
    y_norm=(y*y).sum(1).sqrt()
    cs=xy/(x_norm*y_norm+1e-8)
    return cs


def main_worker(val_loader, args):
    # create model
    model=vgg.vgg16(pretrained=True).cuda()
    model_As, model_Bs, feature_size=prepare_model(model)

    import ADGT
    adgt = ADGT.ADGT(use_cuda=True, name='ImageNet')
    if DATASET=='train':
        loader=None
    else:
        loader=val_loader
    pth = os.path.join(ROOT, 'Split_Model')
    if not os.path.exists(pth):
        os.mkdir(pth)

    cos_sim=np.zeros([9,len(model_As)])
    if True:
        end = time.time()

        for i, (images, target) in enumerate(loader):
            images, target = images.cuda(), target.cuda()
            for j in range(images.size(0)):
                base_img = images[j].unsqueeze(0)
                now = i * BATCH + j
                if now == 26946 or now == 48510:
                    print(now,i,j)
                    img_pth = os.path.join(pth, str(now))
                    if not os.path.exists(img_pth):
                        os.mkdir(img_pth)

                    result=explain_middle_layer_and_raw(base_img,model,model_As,model_Bs,feature_size,method)
                    visualize(result,os.path.join(img_pth, 'baseline.png'),layer_num=len(model_As)+1)
                    save_images2(base_img.detach().cpu().numpy(), os.path.join(img_pth, 'raw.png'))

def visualize(X,save_pth,size=224):
    fig = plt.figure()
    T=len(X)
    N=len(X[0])
    for i in range(N):
        for j in range(T):
            temp=X[j][i]
            #temp=temp.unsqueeze(0)
            temp = F.interpolate(temp, size=(size,size),
                                 mode='bilinear')
            temp=temp.sum(1).squeeze().cpu().numpy()
            plt.subplot(T, N, j*N+i+1)
            ax = fig.add_subplot(T, N, j*N+i+1)
            ax.set_xticks([])
            ax.set_yticks([])
            img = temp
            plt.imshow(img, cmap="seismic")
            high = np.abs(temp).max()
            plt.clim(-high, high)

    plt.gcf().set_size_inches(size*N / 100,  size*T/ 100)
    plt.subplots_adjust(top=0.99, bottom=0.01, right=0.99, left=0.01, hspace=0.05, wspace=0.05)
    plt.savefig(save_pth)
    plt.close()

def visualize2(X,save_pth,size=224):
    fig = plt.figure()
    T=len(X)
    for i in range(T):
        temp=F.relu(X[i])
        #temp=temp.unsqueeze(0)
        if temp.size(2)<size:
            temp = F.interpolate(temp, size=(size,size),
                                 mode='bilinear')
        temp=temp.sum(1).squeeze().cpu().numpy()
        plt.subplot(1,T, i+1)
        ax = fig.add_subplot(1,T, i+1)
        ax.set_xticks([])
        ax.set_yticks([])
        img = temp
        #plt.imshow(img, cmap="seismic")
        plt.imshow(img, cmap="jet")
        #high = np.abs(temp).max()
        #plt.clim(-high, high)

    plt.gcf().set_size_inches(size*T / 100,  size/ 100)
    plt.subplots_adjust(top=0.99, bottom=0.01, right=0.99, left=0.01, hspace=0.05, wspace=0.05)
    plt.savefig(save_pth)
    plt.close()

def visualize_most(X,save_dir,alpha=0,prefix='',methods=()):
    for k,x in enumerate(X):
        a,ts=combine(x,alpha=alpha,mode='most')
        save_images5(a.cpu().numpy(),os.path.join(save_dir,prefix+methods[k]+'_most'+'.png'))
        draw(ts, os.path.join(save_dir, prefix + methods[k] + '_ts' + '.png'))
        draw(x, os.path.join(save_dir, prefix + methods[k] + '_x' + '.png'))

def visualize_most_alpha(X,save_dir,alphas=(0),prefix='',method='DeepLIFT',plot=True):
    #if isinstance(X,list):
    #    x=X
    #else:
    #    x=X[0]
    result=[]
    for k,t in enumerate(alphas):
        a,ts=combine(X,alpha=t,mode='most')
        #save_images5(a.cpu().numpy(),os.path.join(save_dir,prefix+method+str(t)+'_most'+'.png'))
        result.append(a)
        if plot:
            draw(ts, os.path.join(save_dir, prefix + method +str(t)+ '_ts' + '.png'))
    if plot:
        draw(result, os.path.join(save_dir, prefix + method + '_result' + '.png'))
        draw(X, os.path.join(save_dir, prefix + method + '_x' + '.png'))
    return result

def draw(X,pth):
    x=[]
    for m in X:
        temp=m
        #temp=F.relu(temp)
        if temp.size(2)<224:
            #temp=F.interpolate(temp,(224,224),mode='nearest')
            temp = F.interpolate(temp, (224, 224),mode='bilinear')
        temp = temp.sum(1,keepdim=True)
        temp=temp.detach().cpu().numpy()
        high = np.abs(temp).max()
        temp=temp/high
        x.append(temp)
    x=np.concatenate(x,0)
    save_images5(x, pth)
def visualize_all(X,save_dir,prefix='',methods=()):
    result_least = []
    result_most=[]
    result_combine=[]
    result_ys=[]
    result_ra=[]
    result_ts=[]
    result_x=[]
    for k,x in enumerate(X):
        a,b,c,d,e,f=combine(x,mode='all')
        save_images5(a.cpu().numpy(),os.path.join(save_dir,prefix+methods[k]+'_least'+'.png'))
        save_images5(b.cpu().numpy(), os.path.join(save_dir, prefix+methods[k] + '_most' + '.png'))
        save_images5(c.cpu().numpy(), os.path.join(save_dir, prefix+methods[k] + '_combine' + '.png'))
        draw(d, os.path.join(save_dir, prefix+methods[k] + '_ys' + '.png'))
        draw(e, os.path.join(save_dir, prefix+methods[k] + '_ra' + '.png'))
        draw(f, os.path.join(save_dir, prefix+methods[k] + '_ts' + '.png'))
        draw(x, os.path.join(save_dir, prefix+methods[k] + '_x' + '.png'))



if __name__ == '__main__':
    DATASET = 'test'  # 'train'#
    ROOT = 'result2'
    BATCH = 16
    if not os.path.exists(ROOT):
        os.mkdir(ROOT)
    # method=['InputXGradient','SmoothGrad','Guided_BackProp','DeepLIFT','IntegratedGradients','FullGrad']#'GradCAM','CAMERAS',
    method = ['InputXGradient', 'InputXSG', 'InputXGBP', 'DeepLIFT', 'IntegratedGradients',
              'FullGrad']  # 'GradCAM','CAMERAS',
    torch.set_num_threads(4)
    os.environ["CUDA_VISIBLE_DEVICES"] = "7"
    model=resnet.resnet50(pretrained=True).cuda()
    print(model)
    #main()