import sys

sys.path.insert(0, "../")

import math
import torch
from transformers import AutoTokenizer, AutoConfig
from inference.modeling_internvl_chat import InternVLChatModel

vision_model_name = 'OpenGVLab/InternVL3-2B'
cache_dir = None

device = 'cuda:0'
def split_model(model_name):
    device_map = {}
    world_size = torch.cuda.device_count()
    config = AutoConfig.from_pretrained(vision_model_name, cache_dir=cache_dir, trust_remote_code=True)
    num_layers = config.llm_config.num_hidden_layers
    # Since the first GPU will be used for ViT, treat it as half a GPU.
    num_layers_per_gpu = math.ceil(num_layers / (world_size - 0.5))
    num_layers_per_gpu = [num_layers_per_gpu] * world_size
    num_layers_per_gpu[0] = math.ceil(num_layers_per_gpu[0] * 0.5)
    layer_cnt = 0
    for i, num_layer in enumerate(num_layers_per_gpu):
        for j in range(num_layer):
            device_map[f'language_model.model.layers.{layer_cnt}'] = i
            layer_cnt += 1
    device_map['vision_model'] = 0
    device_map['mlp1'] = 0
    device_map['language_model.model.tok_embeddings'] = 0
    device_map['language_model.model.embed_tokens'] = 0
    device_map['language_model.output'] = 0
    device_map['language_model.model.norm'] = 0
    device_map['language_model.model.rotary_emb'] = 0
    device_map['language_model.lm_head'] = 0
    device_map[f'language_model.model.layers.{num_layers - 1}'] = 0

    return device_map


device_map = split_model('InternVL3-2B')
llm = InternVLChatModel.from_pretrained(
    vision_model_name,
    cache_dir=cache_dir,
    torch_dtype=torch.bfloat16,
    low_cpu_mem_usage=True,
    use_flash_attn=True,
    trust_remote_code=True,
    device_map=device_map).eval()
tokenizer = AutoTokenizer.from_pretrained(vision_model_name, cache_dir=cache_dir, trust_remote_code=True, use_fast=False)


import math
import numpy as np
import torch
import torchvision.transforms as T
from decord import VideoReader, cpu
from PIL import Image
from torchvision.transforms.functional import InterpolationMode
from transformers import AutoModel, AutoTokenizer

IMAGENET_MEAN = (0.485, 0.456, 0.406)
IMAGENET_STD = (0.229, 0.224, 0.225)

def build_transform(input_size):
    MEAN, STD = IMAGENET_MEAN, IMAGENET_STD
    transform = T.Compose([
        T.Lambda(lambda img: img.convert('RGB') if img.mode != 'RGB' else img),
        T.Resize((input_size, input_size), interpolation=InterpolationMode.BICUBIC),
        T.ToTensor(),
        T.Normalize(mean=MEAN, std=STD)
    ])
    return transform

def find_closest_aspect_ratio(aspect_ratio, target_ratios, width, height, image_size):
    best_ratio_diff = float('inf')
    best_ratio = (1, 1)
    area = width * height
    for ratio in target_ratios:
        target_aspect_ratio = ratio[0] / ratio[1]
        ratio_diff = abs(aspect_ratio - target_aspect_ratio)
        if ratio_diff < best_ratio_diff:
            best_ratio_diff = ratio_diff
            best_ratio = ratio
        elif ratio_diff == best_ratio_diff:
            if area > 0.5 * image_size * image_size * ratio[0] * ratio[1]:
                best_ratio = ratio
    return best_ratio

def dynamic_preprocess(image, min_num=1, max_num=12, image_size=448, use_thumbnail=False):
    orig_width, orig_height = image.size
    aspect_ratio = orig_width / orig_height

    # calculate the existing image aspect ratio
    target_ratios = set(
        (i, j) for n in range(min_num, max_num + 1) for i in range(1, n + 1) for j in range(1, n + 1) if
        i * j <= max_num and i * j >= min_num)
    target_ratios = sorted(target_ratios, key=lambda x: x[0] * x[1])

    # find the closest aspect ratio to the target
    target_aspect_ratio = find_closest_aspect_ratio(
        aspect_ratio, target_ratios, orig_width, orig_height, image_size)

    # calculate the target width and height
    target_width = image_size * target_aspect_ratio[0]
    target_height = image_size * target_aspect_ratio[1]
    blocks = target_aspect_ratio[0] * target_aspect_ratio[1]

    # resize the image
    resized_img = image.resize((target_width, target_height))
    processed_images = []
    for i in range(blocks):
        box = (
            (i % (target_width // image_size)) * image_size,
            (i // (target_width // image_size)) * image_size,
            ((i % (target_width // image_size)) + 1) * image_size,
            ((i // (target_width // image_size)) + 1) * image_size
        )
        # split the image
        split_img = resized_img.crop(box)
        processed_images.append(split_img)
    assert len(processed_images) == blocks
    if use_thumbnail and len(processed_images) != 1:
        thumbnail_img = image.resize((image_size, image_size))
        processed_images.append(thumbnail_img)
    return processed_images

def load_image(image_file, input_size=448, max_num=12):
    image = Image.open(image_file).convert('RGB')
    transform = build_transform(input_size=input_size)
    images = dynamic_preprocess(image, image_size=input_size, use_thumbnail=True, max_num=max_num)
    pixel_values = [transform(image) for image in images]
    pixel_values = torch.stack(pixel_values)
    return pixel_values

def load_from_pil(image, input_size=448, max_num=12, use_thumbnail=True):
    transform = build_transform(input_size=input_size)
    images = dynamic_preprocess(image, image_size=input_size, use_thumbnail=use_thumbnail, max_num=max_num)
    pixel_values = [transform(image) for image in images]
    pixel_values = torch.stack(pixel_values)
    return pixel_values


@torch.no_grad()
def extract_feature(llm, pixel_values):
    if llm.select_layer == -1:
        vit_embeds = llm.vision_model(
            pixel_values=pixel_values,
            output_hidden_states=False,
            return_dict=True).last_hidden_state
    else:
        vit_embeds = llm.vision_model(
            pixel_values=pixel_values,
            output_hidden_states=True,
            return_dict=True).hidden_states[llm.select_layer]
    vit_embeds = vit_embeds[:, 1:, :]

    h = w = int(vit_embeds.shape[1] ** 0.5)
    vit_embeds = vit_embeds.reshape(vit_embeds.shape[0], h, w, -1)
    vit_embeds = llm.pixel_shuffle(vit_embeds, scale_factor=llm.downsample_ratio)
    vit_embeds = vit_embeds.reshape(vit_embeds.shape[0], -1, vit_embeds.shape[-1])
    vit_embeds = llm.mlp1(vit_embeds)
    vit_embeds = vit_embeds.reshape(vit_embeds.shape[0], h // 2, w // 2, -1)
    return vit_embeds


import os
import json
from tqdm import tqdm
from PIL import Image

num_patches_per_side = llm.config.vision_config.image_size // llm.config.vision_config.patch_size

datasets_dir = '../../../data/'
batch_size = 16

import argparse
parser = argparse.ArgumentParser()
parser.add_argument("--mode", type=str, default='train', choices=['train', 'val'])
parser.add_argument("--limit_samples", type=int, default=None)
args = parser.parse_args()
mode = args.mode
limit_samples = args.limit_samples

vision_model_name_for_path = '-'.join(vision_model_name.split('/'))
train_max_images = [200000]
val_max_images = [5000]

if limit_samples is not None:
    train_max_images = [limit_samples]
    val_max_images = [limit_samples]

# train_max_images = [10]
train_images_dir = [
    '../../../data/coco_subsets/train2017',
    # '../../../data/docvqa/train/documents',
    # '../../../data/chart_qa_images'
]
train_features_dir = f'{datasets_dir}/{vision_model_name_for_path}_mlp/tensors'
train_features_json = f'{datasets_dir}/{vision_model_name_for_path}_mlp/map.json'

print("Train datasets json: ", train_features_json)

# val_max_images = [10]
val_images_dir = [
    '../../../data/coco_subsets/val2017',
    # add another two datasets for full reproducibility
    # '../../../data/docvqa/train/documents',
    # '../../../data/chart_qa_images'
]
val_features_dir = f'{datasets_dir}/{vision_model_name_for_path}_mlp/tensors_val'
val_features_json = f'{datasets_dir}/{vision_model_name_for_path}_mlp/map_val.json'

print("Val datasets json: ", val_features_json)

feature_image_map = {}

if mode == 'train':
    print("Generating train features")
    max_images_list = train_max_images
    images_dir_list = train_images_dir
    features_dir = train_features_dir
    features_json = train_features_json
    image_names_list = [os.listdir(images_dir)[:max_images] for images_dir, max_images in zip(images_dir_list, max_images_list)]
    os.makedirs(features_dir, mode=0o777, exist_ok=True)
elif mode == 'val':
    print("Generating val features")
    max_images_list = val_max_images
    images_dir_list = val_images_dir
    features_dir = val_features_dir
    features_json = val_features_json
    image_names_list = [os.listdir(images_dir)[-max_images:] for images_dir, max_images in zip(images_dir_list, max_images_list)]
    os.makedirs(features_dir, mode=0o777, exist_ok=True)
else:
    raise Exception

with torch.inference_mode(), torch.no_grad():
    for (image_names, images_dir) in zip(image_names_list, images_dir_list):
        for i in tqdm(range(0, len(image_names), batch_size)):
            batch_image_names = image_names[i:i+batch_size]
            batch_processed_images = []
            batch_image_paths = []
            batch_feature_paths = []
            batch_crop_numbers = []

            for image_name in batch_image_names:
                feature_name = image_name.split('.')[0]
                image_path = os.path.join(images_dir, image_name)

                try:
                    example = Image.open(image_path).convert('RGB')
                    pixel_values = load_from_pil(example).to(torch.bfloat16)

                    number = np.random.randint(0, len(pixel_values))
                    batch_processed_images.append(pixel_values[number])
                    batch_image_paths.append(image_path)
                    batch_feature_paths.append(os.path.join(features_dir, f'{feature_name}_{number}.pt'))
                    batch_crop_numbers.append(number)

                    # pixel_values = [pv for pv in pixel_values]
                    # batch_processed_images.extend(pixel_values)
                    # batch_image_paths.extend([image_path] * len(pixel_values))
                    # batch_feature_paths.extend([
                    #     os.path.join(features_dir, f'{feature_name}_{i}.pt') for i in range(len(pixel_values))
                    #     ])
                    # batch_crop_numbers.extend([i for i in range(len(pixel_values))])
                except Exception as e:
                    print(f"Error processing image {image_path}: {e}")
                    continue # Skip this image if loading/preprocessing fails

            if not batch_processed_images: # If all images in batch failed
                continue

            images_batch = torch.stack(batch_processed_images).to(device)
            batch_features: torch.Tensor = extract_feature(llm, images_batch)

            assert batch_features.dtype == torch.bfloat16

            # Process and save features for each image in the batch
            for idx in range(batch_features.shape[0]):
                image_path = batch_image_paths[idx]
                feature_path = batch_feature_paths[idx]
                number = batch_crop_numbers[idx]
                feature_image_map[feature_path] = (image_path, number)

                features = batch_features[idx]
                features_reshaped = features.clone()
                torch.save(features_reshaped, feature_path)

    with open(features_json, 'w') as config:
        json.dump(feature_image_map, config)