# --------------------------------------------------------
# InternVL
# Copyright (c) 2024 OpenGVLab
# Licensed under The MIT License [see LICENSE for details]
# --------------------------------------------------------

import warnings
from typing import List, Optional, Tuple, Union

import torch.utils.checkpoint
import transformers
import numpy as np

from torch import nn
from torch.nn import CrossEntropyLoss
from transformers import (AutoModel, GenerationConfig, LlamaForCausalLM)
from transformers.modeling_outputs import CausalLMOutputWithPast
from transformers.modeling_utils import PreTrainedModel
from transformers.utils import ModelOutput, logging

from .modeling_qwen2 import Qwen2ForCausalLM
from .configuration_internvl_chat import InternVLChatConfig
from .conversation import get_conv_template
from .modeling_intern_vit import InternVisionModel, has_flash_attn

from .sampler import GumbelSampler

logger = logging.get_logger(__name__)


def version_cmp(v1, v2, op='eq'):
    import operator

    from packaging import version
    op_func = getattr(operator, op)
    return op_func(version.parse(v1), version.parse(v2))


class InternVLChatModel(PreTrainedModel):
    config_class = InternVLChatConfig
    main_input_name = 'pixel_values'
    base_model_prefix = 'language_model'
    _supports_flash_attn_2 = True
    supports_gradient_checkpointing = True
    _no_split_modules = ['InternVisionModel', 'LlamaDecoderLayer', 'Qwen2DecoderLayer']

    def __init__(self, config: InternVLChatConfig, vision_model=None, language_model=None, use_flash_attn=True):
        super().__init__(config)

        self._tp_plan = {}

        assert version_cmp(transformers.__version__, '4.37.0', 'ge')
        image_size = config.force_image_size or config.vision_config.image_size
        patch_size = config.vision_config.patch_size
        self.patch_size = patch_size
        self.select_layer = config.select_layer
        self.template = config.template
        self.num_image_token = int((image_size // patch_size) ** 2 * (config.downsample_ratio ** 2))
        self.downsample_ratio = config.downsample_ratio
        self.ps_version = config.ps_version
        use_flash_attn = use_flash_attn if has_flash_attn else False
        config.vision_config.use_flash_attn = True if use_flash_attn else False
        config.llm_config._attn_implementation = 'flash_attention_2' if use_flash_attn else 'eager'

        logger.info(f'num_image_token: {self.num_image_token}')
        logger.info(f'ps_version: {self.ps_version}')
        if vision_model is not None:
            self.vision_model = vision_model
        else:
            self.vision_model = InternVisionModel(config.vision_config)
        if language_model is not None:
            self.language_model = language_model
        else:
            if config.llm_config.architectures[0] == 'LlamaForCausalLM':
                self.language_model = LlamaForCausalLM(config.llm_config)
            elif config.llm_config.architectures[0] == 'Qwen2ForCausalLM':
                self.language_model = Qwen2ForCausalLM(config.llm_config)
            else:
                raise NotImplementedError(f'{config.llm_config.architectures[0]} is not implemented.')

        vit_hidden_size = config.vision_config.hidden_size
        llm_hidden_size = config.llm_config.hidden_size

        self.mlp1 = nn.Sequential(
            nn.LayerNorm(vit_hidden_size * int(1 / self.downsample_ratio) ** 2),
            nn.Linear(vit_hidden_size * int(1 / self.downsample_ratio) ** 2, llm_hidden_size),
            nn.GELU(),
            nn.Linear(llm_hidden_size, llm_hidden_size)
        )

        self.img_context_token_id = None
        self.conv_template = get_conv_template(self.template)
        self.system_message = self.conv_template.system_message

    def load_sampler(self, encoder_layers, config_pretrained_path, config_cache_dir, sampler_weights=None, 
                     sampler_type=None, iva_factor=None, th=None):
        self.sampler = GumbelSampler(encoder_layers, config_pretrained_path, config_cache_dir, 
                                     sampler_type, iva_factor, th).to(self.device)
        if sampler_weights is not None:
            self.sampler.load_state_dict(torch.load(sampler_weights, weights_only=True), strict=False)
        self.sampler.eval()
        print(f'GumbelSampler[{sampler_type}] is loaded.')

    def forward(
            self,
            pixel_values: torch.FloatTensor,
            input_ids: torch.LongTensor = None,
            attention_mask: Optional[torch.Tensor] = None,
            position_ids: Optional[torch.LongTensor] = None,
            image_flags: Optional[torch.LongTensor] = None,
            past_key_values: Optional[List[torch.FloatTensor]] = None,
            labels: Optional[torch.LongTensor] = None,
            use_cache: Optional[bool] = None,
            output_attentions: Optional[bool] = None,
            output_hidden_states: Optional[bool] = None,
            return_dict: Optional[bool] = None,
    ) -> Union[Tuple, CausalLMOutputWithPast]:
        return_dict = return_dict if return_dict is not None else self.config.use_return_dict

        image_flags = image_flags.squeeze(-1)
        input_embeds = self.language_model.get_input_embeddings()(input_ids).clone()

        vit_embeds = self.extract_feature(pixel_values)
        vit_embeds = vit_embeds[image_flags == 1]
        vit_batch_size = pixel_values.shape[0]

        B, N, C = input_embeds.shape
        input_embeds = input_embeds.reshape(B * N, C)

        if torch.distributed.is_initialized() and torch.distributed.get_rank() == 0:
            print(f'dynamic ViT batch size: {vit_batch_size}, images per sample: {vit_batch_size / B}, dynamic token length: {N}')

        input_ids = input_ids.reshape(B * N)
        selected = (input_ids == self.img_context_token_id)
        try:
            input_embeds[selected] = input_embeds[selected] * 0.0 + vit_embeds.reshape(-1, C)
        except Exception as e:
            vit_embeds = vit_embeds.reshape(-1, C)
            print(f'warning: {e}, input_embeds[selected].shape={input_embeds[selected].shape}, '
                  f'vit_embeds.shape={vit_embeds.shape}')
            n_token = min(selected.sum(), vit_embeds.size(0))
            input_embeds[selected][:n_token] = input_embeds[selected][:n_token] * 0.0 + vit_embeds[:n_token]

        input_embeds = input_embeds.reshape(B, N, C)

        outputs = self.language_model(
            inputs_embeds=input_embeds,
            attention_mask=attention_mask,
            position_ids=position_ids,
            past_key_values=past_key_values,
            use_cache=use_cache,
            output_attentions=output_attentions,
            output_hidden_states=output_hidden_states,
            return_dict=return_dict,
        )
        logits = outputs.logits

        loss = None
        if labels is not None:
            # Shift so that tokens < n predict n
            shift_logits = logits[..., :-1, :].contiguous()
            shift_labels = labels[..., 1:].contiguous()
            # Flatten the tokens
            loss_fct = CrossEntropyLoss()
            shift_logits = shift_logits.view(-1, self.language_model.config.vocab_size)
            shift_labels = shift_labels.view(-1)
            # Enable model parallelism
            shift_labels = shift_labels.to(shift_logits.device)
            loss = loss_fct(shift_logits, shift_labels)

        if not return_dict:
            output = (logits,) + outputs[1:]
            return (loss,) + output if loss is not None else output

        return CausalLMOutputWithPast(
            loss=loss,
            logits=logits,
            past_key_values=outputs.past_key_values,
            hidden_states=outputs.hidden_states,
            attentions=outputs.attentions,
        )

    def pixel_shuffle(self, x, scale_factor=0.5):
        n, w, h, c = x.size()
        # N, W, H, C --> N, W, H * scale, C // scale
        x = x.view(n, w, int(h * scale_factor), int(c / scale_factor))
        # N, W, H * scale, C // scale --> N, H * scale, W, C // scale
        x = x.permute(0, 2, 1, 3).contiguous()
        # N, H * scale, W, C // scale --> N, H * scale, W * scale, C // (scale ** 2)
        x = x.view(n, int(h * scale_factor), int(w * scale_factor),
                   int(c / (scale_factor * scale_factor)))
        if self.ps_version == 'v1':
            warnings.warn("In ps_version 'v1', the height and width have not been swapped back, "
                          'which results in a transposed image.')
        else:
            x = x.permute(0, 2, 1, 3).contiguous()
        return x

    def extract_feature(self, pixel_values):
        if self.select_layer == -1:
            vit_embeds = self.vision_model(
                pixel_values=pixel_values,
                output_hidden_states=False,
                return_dict=True).last_hidden_state
        else:
            vit_embeds = self.vision_model(
                pixel_values=pixel_values,
                output_hidden_states=True,
                return_dict=True).hidden_states[self.select_layer]
        vit_embeds = vit_embeds[:, 1:, :]

        h = w = int(vit_embeds.shape[1] ** 0.5)
        vit_embeds = vit_embeds.reshape(vit_embeds.shape[0], h, w, -1)
        vit_embeds = self.pixel_shuffle(vit_embeds, scale_factor=self.downsample_ratio)
        vit_embeds = vit_embeds.reshape(vit_embeds.shape[0], -1, vit_embeds.shape[-1])
        vit_embeds = self.mlp1(vit_embeds)
        return vit_embeds

    def batch_chat(self, tokenizer, pixel_values, questions, generation_config, num_patches_list=None,
                   history=None, return_history=False, IMG_START_TOKEN='<img>', IMG_END_TOKEN='</img>',
                   IMG_CONTEXT_TOKEN='<IMG_CONTEXT>', verbose=False, image_counts=None):
        if history is not None or return_history:
            print('Now multi-turn chat is not supported in batch_chat.')
            raise NotImplementedError

        if image_counts is not None:
            num_patches_list = image_counts
            print('Warning: `image_counts` is deprecated. Please use `num_patches_list` instead.')

        img_context_token_id = tokenizer.convert_tokens_to_ids(IMG_CONTEXT_TOKEN)
        self.img_context_token_id = img_context_token_id

        if verbose and pixel_values is not None:
            image_bs = pixel_values.shape[0]
            print(f'dynamic ViT batch size: {image_bs}')

        queries = []
        for idx, num_patches in enumerate(num_patches_list):
            question = questions[idx]
            if pixel_values is not None and '<image>' not in question:
                question = '<image>\n' + question
            template = get_conv_template(self.template)
            template.system_message = self.system_message
            template.append_message(template.roles[0], question)
            template.append_message(template.roles[1], None)
            query = template.get_prompt()

            image_tokens = IMG_START_TOKEN + IMG_CONTEXT_TOKEN * self.num_image_token * num_patches + IMG_END_TOKEN
            query = query.replace('<image>', image_tokens, 1)
            queries.append(query)

        tokenizer.padding_side = 'left'
        model_inputs = tokenizer(queries, return_tensors='pt', padding=True)
        input_ids = model_inputs['input_ids'].to(self.device)
        attention_mask = model_inputs['attention_mask'].to(self.device)
        eos_token_id = tokenizer.convert_tokens_to_ids(template.sep.strip())
        generation_config['eos_token_id'] = eos_token_id
        generation_output = self.generate(
            pixel_values=pixel_values,
            input_ids=input_ids,
            attention_mask=attention_mask,
            **generation_config
        )
        responses = tokenizer.batch_decode(generation_output, skip_special_tokens=True)
        responses = [response.split(template.sep.strip())[0].strip() for response in responses]
        return responses

    def chat(self, tokenizer, pixel_values, question, generation_config, 
             num_patches_list=None,
             target_aspect_ratio_list = None,
             keep_small_image: bool = None,
             keep_big_image: bool = None,
             draw_hist: bool = False, 
             draw_small: bool = False,
             draw_big: bool = False,
             history=None, return_history=False,
             IMG_START_TOKEN='<img>', IMG_END_TOKEN='</img>', IMG_CONTEXT_TOKEN='<IMG_CONTEXT>',
             verbose=False):

        if history is None and pixel_values is not None and '<image>' not in question:
            question = '<image>\n' + question

        if num_patches_list is None:
            num_patches_list = [pixel_values.shape[0]] if pixel_values is not None else []
        assert pixel_values is None or len(pixel_values) == sum(num_patches_list)

        img_context_token_id = tokenizer.convert_tokens_to_ids(IMG_CONTEXT_TOKEN)
        self.img_context_token_id = img_context_token_id

        template = get_conv_template(self.template)
        template.system_message = self.system_message
        eos_token_id = tokenizer.convert_tokens_to_ids(template.sep.strip())

        history = [] if history is None else history
        for (old_question, old_answer) in history:
            template.append_message(template.roles[0], old_question)
            template.append_message(template.roles[1], old_answer)
        template.append_message(template.roles[0], question)
        template.append_message(template.roles[1], None)
        query = template.get_prompt()

        if verbose and pixel_values is not None:
            image_bs = pixel_values.shape[0]
            print(f'dynamic ViT batch size: {image_bs}')

        for num_patches in num_patches_list:
            image_tokens = IMG_START_TOKEN + IMG_CONTEXT_TOKEN * self.num_image_token * num_patches + IMG_END_TOKEN
            query = query.replace('<image>', image_tokens, 1)

        model_inputs = tokenizer(query, return_tensors='pt')
        input_ids = model_inputs['input_ids'].to(self.device)
        attention_mask = model_inputs['attention_mask'].to(self.device)
        generation_config['eos_token_id'] = eos_token_id
        generation_output, (selected_tokens_number, all_tokens_number) = self.generate(
            pixel_values=pixel_values,
            input_ids=input_ids,
            attention_mask=attention_mask,
            num_patches_list=num_patches_list,
            target_aspect_ratio_list=target_aspect_ratio_list,
            keep_small_image=keep_small_image,
            keep_big_image=keep_big_image,
            draw_hist=draw_hist,
            draw_small=draw_small,
            draw_big=draw_big,
            **generation_config
        )
        response = tokenizer.batch_decode(generation_output, skip_special_tokens=True)[0]
        response = response.split(template.sep.strip())[0].strip()
        history.append((question, response))
        if return_history:
            return response, history, (selected_tokens_number, all_tokens_number)
        else:
            query_to_print = query.replace(IMG_CONTEXT_TOKEN, '')
            query_to_print = query_to_print.replace(f'{IMG_START_TOKEN}{IMG_END_TOKEN}', '<image>')
            if verbose:
                print(query_to_print, response)
            return response, (selected_tokens_number, all_tokens_number)

    def find_starts_ends(self, mask, position_ids):
        start_mask = torch.cat([mask[:1] * False, mask])
        start_mask = start_mask[1:] * ~start_mask[:-1]
        starts = position_ids[start_mask]

        ends_mask = torch.cat([mask, mask[:1] * False])
        ends_mask = ends_mask[:-1] * ~ends_mask[1:]
        ends = position_ids[ends_mask]
        
        return torch.stack([starts, ends], dim=0).t()

    def replace_masks(self, mask, sides_of_segments, gumbel_masks):
        gumbel_mask_i = 0
        for segment_i, (segment_start, segment_end) in enumerate(sides_of_segments):
            current_segment_start, current_segment_end = segment_start, segment_start + gumbel_masks.shape[-1]
            
            while current_segment_start <= segment_end:
                mask[current_segment_start:current_segment_end] = gumbel_masks[gumbel_mask_i]
                current_segment_start = current_segment_end
                current_segment_end = current_segment_end + gumbel_masks.shape[-1]
                gumbel_mask_i += 1

        return mask

    def prepare_image(self, gumbel_mask, logits, target_aspect_ratio, pixel_values):
        from torch.nn import functional as F
        mask_side = int(gumbel_mask.shape[1]**0.5)
        image_side = pixel_values.shape[-1]
        target_aspect_ratio = target_aspect_ratio[::-1]
        minus_one = -1 if pixel_values.shape[0] > 1 else 1


        # ------------- IMAGE -------------
        small_image = pixel_values[-1].float() # [3, 448, 448]
        big_image = pixel_values[:minus_one].reshape(target_aspect_ratio[0], target_aspect_ratio[1],
                                              pixel_values.shape[1], pixel_values.shape[2], 
                                              pixel_values.shape[3]).float() # [2, 5, 3, 448, 448]
        small_image = small_image.permute(1, 2, 0).cpu() # [448, 448, 3]
        big_image = big_image.permute(0, 1, 3, 4, 2).cpu() # [2, 5, 448, 448, 3]
        # --------------- GUMBEL MASK ---------------
        gms = gumbel_mask.reshape(gumbel_mask.shape[0], 1, mask_side, mask_side).float()
        gms = gms.repeat(1, 3, 1, 1)
        gms = F.interpolate(gms, size=(image_side, image_side)) > 0

        small_mask = gms[-1] # [3, 448, 448]
        big_mask = gms[:minus_one] # [10, 3, 448, 448]
        big_mask = big_mask.reshape(target_aspect_ratio[0], target_aspect_ratio[1],
                                    big_mask.shape[1], big_mask.shape[2], big_mask.shape[3])
                
        small_mask = small_mask.permute(1, 2, 0).cpu().to(torch.float32) # [448, 448, 3]
        big_mask = big_mask.permute(0, 1, 3, 4, 2).cpu().to(torch.float32) # [2, 5, 448, 448, 3]
        # --------------- LOGITS ----------------
        ls = logits.reshape(logits.shape[0], 1, mask_side, mask_side).float()
        ls = ls.repeat(1, 3, 1, 1)
        ls = F.interpolate(ls, size=(image_side, image_side))

        small_ls = ls[-1] # [3, 448, 448]
        big_ls = ls[:minus_one] # [10, 3, 448, 448]
        big_ls = big_ls.reshape(target_aspect_ratio[0], target_aspect_ratio[1],
                                    big_ls.shape[1], big_ls.shape[2], big_ls.shape[3])
                
        small_ls = small_ls.permute(1, 2, 0).cpu().to(torch.float32) # [448, 448, 3]
        big_ls = big_ls.permute(0, 1, 3, 4, 2).cpu().to(torch.float32) # [2, 5, 448, 448, 3]
        # --------------- BIG TRANSFORM ----------------
        big_image = big_image.permute(0, 2, 1, 3, 4).reshape(big_image.shape[0] * big_image.shape[2], 
                                                             big_image.shape[1] * big_image.shape[3], 
                                                             big_image.shape[4])
        big_mask = big_mask.permute(0, 2, 1, 3, 4).reshape(big_mask.shape[0] * big_mask.shape[2], 
                                                           big_mask.shape[1] * big_mask.shape[3], 
                                                           big_mask.shape[4])
        big_ls = big_ls.permute(0, 2, 1, 3, 4).reshape(big_ls.shape[0] * big_ls.shape[2], 
                                                           big_ls.shape[1] * big_ls.shape[3], 
                                                           big_ls.shape[4])
        # -------------------------------
        big_image = (big_image - big_image.min()) / (big_image.max() - big_image.min())
        small_image = (small_image - small_image.min()) / (small_image.max() - small_image.min())
        return (small_image, small_ls, small_mask), (big_image, big_ls, big_mask)

    def draw_video(self, 
                   gumbel_mask, logits, num_patches_list, target_aspect_ratio_list, pixel_values,
                   draw_hist=False, draw_small=False, draw_big=False):
        from math import ceil
        import matplotlib.pyplot as plt

        gumbel_mask_list = torch.split(gumbel_mask, num_patches_list, dim=0)
        logits_list = torch.split(logits, num_patches_list, dim=0)
        pixel_values_list = torch.split(pixel_values, num_patches_list, dim=0)

        small_info_list, big_info_list = [], []
        for gm, l, tar, pv in zip(gumbel_mask_list, logits_list, target_aspect_ratio_list, pixel_values_list):
            small_info, big_info = self.prepare_image(gm, l, tar, pv)
            small_info_list.append(small_info)
            big_info_list.append(big_info)

        if draw_hist:
            plt.figure(figsize=(5, 5))
            plt.hist(logits.flatten().cpu().detach(), bins=100, density=True)
            plt.plot([self.sampler.th, self.sampler.th], [0, 10])
            plt.show()

        if draw_small:
            ncols, nrows, scale = ceil(len(small_info_list)**0.5), ceil(len(small_info_list)**0.5), 3
            if ncols == nrows == 1: scale = 5
            fig, axes = plt.subplots(ncols=ncols, nrows=nrows, figsize=(ncols * scale, nrows * scale))
            if ncols == nrows == 1: axes = np.array([[axes]])
            elif ncols == 1: axes = axes[None]
            elif nrows == 1: axes = axes[:, None]

            for i, (small_info, gm) in enumerate(zip(small_info_list, gumbel_mask_list)):
                im, ls, m = small_info
                small_image = (im - im.min()) / (im.max() - im.min())
                small_image[m[..., 0] < 0.5] = torch.tensor([135, 206, 235]) / 255

                axes[i // ncols, i % ncols].imshow(small_image)
                axes[i // ncols, i % ncols].set_title(
                    f'tokens number: {gm[-1].sum().item()}\nshare: {round(gm[-1].float().mean().item(), 2)}')
                axes[i // ncols, i % ncols].axis('off')
            plt.show()

        if draw_big:
            ncols, nrows, scale = ceil(len(big_info_list)**0.5), ceil(len(big_info_list)**0.5), 3
            if ncols == nrows == 1: scale = 10
            fig, axes = plt.subplots(ncols=ncols, nrows=nrows, figsize=(ncols * scale, nrows * scale))
            if ncols == nrows == 1: axes = np.array([[axes]])
            elif ncols == 1: axes = axes[None]
            elif nrows == 1: axes = axes[:, None]

            for i, (big_info, gm) in enumerate(zip(big_info_list, gumbel_mask_list)):
                im, ls, m = big_info
                small_image = (im - im.min()) / (im.max() - im.min())
                small_image[m[..., 0] < 0.5] = torch.tensor([135, 206, 235]) / 255

                axes[i // ncols, i % ncols].imshow(small_image)
                axes[i // ncols, i % ncols].set_title(
                    f'tokens number: {gm.sum().item()}\nshare: {round(gm.float().mean().item(), 2)}')
                axes[i // ncols, i % ncols].axis('off')
            plt.show()


    @torch.no_grad()
    def generate(
            self,
            pixel_values: Optional[torch.FloatTensor] = None,
            input_ids: Optional[torch.FloatTensor] = None,
            attention_mask: Optional[torch.LongTensor] = None,
            visual_features: Optional[torch.FloatTensor] = None,
            num_patches_list: Optional[list[int]] = None,
            target_aspect_ratio_list: Optional[list] = None,
            keep_small_image: bool = None,
            draw_hist: bool = False, 
            draw_small: bool = False,
            draw_big: bool = False,
            generation_config: Optional[GenerationConfig] = None,
            output_hidden_states: Optional[bool] = None,
            **generate_kwargs,
    ) -> torch.LongTensor:
        assert input_ids.shape[0] == 1

        assert self.img_context_token_id is not None
        if pixel_values is not None:
            if visual_features is not None:
                vit_embeds = visual_features
            else:
                vit_embeds = self.extract_feature(pixel_values)

            keep_list = [None for _ in range(vit_embeds.shape[0])]
            for i in np.cumsum(num_patches_list):
                keep_list[i - 1] = keep_small_image

            val_dict = self.sampler(vit_embeds, keep_list)
            gumbel_mask = val_dict['gumbel_mask']
            logits = torch.sigmoid(val_dict['logits'][..., 0])

            selected_tokens_number = gumbel_mask.sum()
            all_tokens_number = np.prod(gumbel_mask.shape)
            #############################################
            #############################################
            #############################################
            if (draw_hist or draw_small or draw_big) and (target_aspect_ratio_list is not None):
                self.draw_video(
                    gumbel_mask, logits, num_patches_list, target_aspect_ratio_list, pixel_values,
                    draw_hist=draw_hist, draw_small=draw_small, draw_big=draw_big)
            #############################################
            #############################################
            #############################################

            input_embeds = self.language_model.get_input_embeddings()(input_ids)
            B, N, C = input_embeds.shape
            input_embeds = input_embeds.reshape(B * N, C)

            input_ids = input_ids.reshape(B * N)
            position_ids = torch.arange(len(input_ids), device=input_ids.device)

            selected = (input_ids == self.img_context_token_id)
            sides_of_segments = self.find_starts_ends(selected, position_ids)
            selected = self.replace_masks(selected, sides_of_segments, gumbel_mask)

            assert sides_of_segments.shape[-1] != 0
            input_embeds[selected] = vit_embeds.reshape(-1, C)[gumbel_mask.flatten()].to(input_embeds.device)

            selected = selected + (input_ids != self.img_context_token_id)
            input_embeds = input_embeds[selected]
            position_ids = position_ids[selected]
            attention_mask = attention_mask[selected[None]]
            
            input_embeds = input_embeds.reshape(B, -1, C)
            position_ids = position_ids[None]
            attention_mask = attention_mask[None]
        else:
            input_embeds = self.language_model.get_input_embeddings()(input_ids)

        outputs = self.language_model.generate(
            inputs_embeds=input_embeds,
            position_ids=position_ids,
            attention_mask=attention_mask,
            generation_config=generation_config,
            output_hidden_states=output_hidden_states,
            use_cache=True,
            **generate_kwargs,
        )

        return outputs, (selected_tokens_number, all_tokens_number)

    @property
    def lm_head(self):
        return self.language_model.get_output_embeddings()

    def get_input_embeddings(self):
        return self.language_model.get_input_embeddings()

    def get_output_embeddings(self):
        return self.language_model.get_output_embeddings()
