import torch
import torch.nn as nn
import torch.nn.functional as F
from layers.SelfAttention_Family import FullAttention, FullAttention_temp
from layers.SWTAttention_Family import WaveletEmbedding
from layers.Embed import PositionalEmbedding
import numpy as np


class PerVarProjectionHead(nn.Module):
    def __init__(self, n_vars, d_model, target_window, dropout=0.0, features='M'):
        super().__init__()
        self.n_vars = n_vars
        self.d_model = d_model
        self.target_window = target_window
        self.dropout = nn.Dropout(dropout)
        self.features = features
        self.heads = nn.ModuleList([nn.Linear(d_model, target_window) for _ in range(n_vars)])
        
    def forward(self, x):  # x: [bs x nvars x d_model x patch_num]
        bs, nvars, d_model, patch_num = x.shape
        out_per_var = []

        if self.features == 'M':
            for i in range(nvars):
                global_token_i = x[:, i, :, -1]
                out_i = self.heads[i](global_token_i)
                out_i = self.dropout(out_i)
                out_per_var.append(out_i)
        elif self.features == 'MS':
            global_token_target_var = x[:, -1, :, -1]
            out_target_var = self.heads[-1](global_token_target_var)
            out_target_var = self.dropout(out_target_var)
            out_per_var.append(out_target_var)

        out = torch.stack(out_per_var, dim=-1) # out: (bs, target_window, nvars)
        
        return out

class SingleProjectionHead(nn.Module):
    def __init__(self, n_vars, nf, target_window, head_dropout=0):
        super().__init__()
        self.n_vars = n_vars
        self.linear = nn.Linear(nf, target_window)
        self.dropout = nn.Dropout(head_dropout)
        self.flatten = nn.Flatten(start_dim=-2)

    def forward(self, x):  # x: [bs x nvars x d_model x patch_num]
        #x = x[:, :, :, -1] # Select Global Token
        #x = self.flatten(x)
        x = self.linear(x)
        x = self.dropout(x)
        x = x.permute(0, 2, 1)
        return x
 
class GlobalPatchEmbedding(nn.Module):
    def __init__(self, n_vars, d_model, patch_len, dropout, M):
        super(GlobalPatchEmbedding, self).__init__()
        self.patch_len = patch_len
        self.d_model = d_model
        self.n_vars = n_vars
        self.dropout = nn.Dropout(dropout)
        self.value_embedding = nn.Linear(patch_len, d_model, bias=False)
        self.position_embedding = PositionalEmbedding(d_model)
        self.glb_token = nn.Parameter(torch.randn(1, n_vars, m, 1, patch_len))  # Global token per channel per scale

    def forward(self, x):  # x: [B, N, M, L]
        B, N, M, L = x.shape
        P = L // self.patch_len  # number of patches
        x = x.unfold(dimension=-1, size=self.patch_len, step=self.patch_len)  # [B, N, M, P, patch_len]

        # Embed value and position: reshape for embedding
        x = x.view(-1, self.patch_len)  # [B*N*M*P, patch_len]
        x = self.value_embedding(x)  # [B*N*M*P, d_model]
        x = x.view(B, N, M, P, -1)
        x = x + self.position_embedding(x)

        # Add global token: [B, N, M, P+1, d_model]
        glb = self.glb_token.repeat(B, 1, 1, 1, 1)
        x = torch.cat([x, glb], dim=3)
        return self.dropout(x), P + 1  # output shape: [B, N, M, P+1, d_model]


class PatchWaveletEmbedding(nn.Module):
    def __init__(self, n_vars, d_model, patch_len, dropout, wv='db1', m=3, requires_grad=True):
        super().__init__()
        self.d_model = d_model
        self.n_vars = n_vars
        self.m = m
        self.wavelet_transform = WaveletEmbedding(d_channel=n_vars, swt=True, requires_grad=requires_grad, wv=wv, m=m)
        self.global_patch_embedding = GlobalPatchEmbedding(n_vars, d_model // (patch_len), patch_len, dropout, M=m+1)
        self.value_embedding = nn.Linear(1, d_model)  # Initial projection before wavelet

    def forward(self, x):  # x: [B, L, N]
        B, L, N = x.shape

        # Step 1: Value Embedding (without global token)
        x = x.permute(0, 2, 1)
        x = self.value_embedding(x)

        # Step 2: Wavelet transform → [B, N, M, L]
        x = self.wavelet_transform(x)

        # Step 3: Patching + global token concat → [B, N, M, P+1, d_model/patch_len]
        x, P_plus_1 = self.global_patch_embedding(x)  # x: [B, N, M, P+1, d_model/patch_len]

        # Final reshape for attention: [B*N, P+1, M, d_model/patch_len]
        x = x.permute(0, 1, 3, 2, 4).reshape(B * N, P_plus_1, self.m + 1, -1)

        return x, N  # N = num channels used later for reshaping

class AttentionLayer(nn.Module):
    def __init__(self, attention, d_model, n_heads, d_keys=None,
                 d_values=None):
        super(AttentionLayer, self).__init__()


        self.inner_attention = attention
        self.query_projection = nn.Linear(d_model, d_model)
        self.key_projection = nn.Linear(d_model, d_model)
        self.value_projection = nn.Linear(d_model, d_model)
        self.out_projection = nn.Linear(d_model, d_model)
        self.n_heads = n_heads
        

    def forward(self, queries, keys, values, attn_mask, tau=None, delta=None):
        B, L, H, E = queries.shape
        _, S, _, _= keys.shape
        H = self.n_heads

        queries = self.query_projection(queries)
        keys = self.key_projection(keys)
        values = self.value_projection(values)
        # attention map 받기 위해서 수정
        out, attn = self.inner_attention(
            queries,
            keys,
            values,
            attn_mask,
            tau=tau,
            delta=delta
        )
        # attention map 안 쓸 때
        # out = self.inner_attention(
        #     queries,
        #     keys,
        #     values,
        #     attn_mask,
        #     tau=tau,
        #     delta=delta
        # )

        return self.out_projection(out), attn

class Encoder(nn.Module):
    def __init__(self, layers, norm_layer=None, projection=None):
        super(Encoder, self).__init__()
        self.layers = nn.ModuleList(layers)
        self.norm = norm_layer
        self.projection = projection

    def forward(self, x, x_mask=None, tau=None, delta=None):
        attn_list = []
        for layer in self.layers:
            # for attention map 쓰는 모델
            x, attn = layer(x, x_mask=x_mask, tau=tau, delta=delta)
            # for attention map 안 쓰는 모델
            # x = layer(x, x_mask=x_mask, tau=tau, delta=delta)
            attn_list.append(attn)

        if self.norm is not None:
            x = self.norm(x)

        if self.projection is not None:
            x = self.projection(x)
        return x, attn_list


class EncoderLayer(nn.Module):
    def __init__(self, self_attention, global_token_attention, d_model, d_ff=None,
                 dropout=0.1, activation="relu", n_vars = 7):
        super(EncoderLayer, self).__init__()
        d_ff = d_ff or 4 * d_model
        self.self_attention = self_attention
        self.global_token_attention = global_token_attention
        self.n_vars = n_vars
        self.linear1 = nn.Linear(d_model, d_ff)
        self.linear2 = nn.Linear(d_ff, d_model)

        self.norm1 = nn.LayerNorm(d_model)
        self.norm2 = nn.LayerNorm(d_model)
        self.norm3 = nn.LayerNorm(d_model)

        self.dropout = nn.Dropout(dropout)
        self.activation = F.relu if activation == "relu" else F.gelu

    def forward(self, x, x_mask=None, tau=None, delta=None):
        #[B*N, m+1, num_patch + 1, d_model]
        B, M, P, D = x.shape
        B = B // self.n_vars
        
        
        temp, _ = self.self_attention(
            x, x, x,
            attn_mask=x_mask,
            tau=tau, delta=None
        )
        
        x = x + self.dropout(temp)
        
        x = self.norm1(x)
        
        x_glb_ori = x[:, :, -1, :].unsqueeze(1)  # [B*N, 1, M, D]

        x_glb = x_glb_ori.reshape(B, -1, M, D)  # [B, N, M, D]
        
        # for attention map 쓰는 모델
        x_glb_attn, attn_glb = self.global_token_attention(
            x_glb, x_glb, x_glb,
            attn_mask=x_mask,
            tau=tau, delta=delta
        )
        
        x_glb_attn = self.dropout(x_glb_attn)
        x_glb_attn = torch.reshape(x_glb_attn,
                                   (x_glb_attn.shape[0] * x_glb_attn.shape[1], 1, M, D))  # [B*N, 1, M, D]
        x_glb = x_glb_ori + x_glb_attn
        x_glb = self.norm2(x_glb).permute(0, 2, 1, 3)
        
        y = x = torch.cat([x[:, :, :-1, :], x_glb], dim=2)  # [B*N, M, P+1, D]
        
        y = self.dropout(self.activation(self.linear1(y)))
        y = self.dropout(self.linear2(y))

        return self.norm3(x + y), attn_glb


class Model(nn.Module):

    def __init__(self, configs):
        super(Model, self).__init__()
        self.task_name = configs.task_name
        self.features = configs.features
        self.seq_len = configs.seq_len
        self.pred_len = configs.pred_len
        self.use_norm = configs.use_norm
        self.patch_len = configs.patch_len
        self.patch_num = int(configs.seq_len // configs.patch_len)
        self.n_vars = 1 if configs.features == 'MS' else configs.enc_in
        # temperature 초기화
        if configs.cross_temp == -1:
            self.cross_temp = nn.Parameter(torch.ones(self.n_vars) * 0.1, requires_grad=True)
        else:
            self.cross_temp = configs.cross_temp
        
        # Embedding
        #self.global_patch_embedding = GlobalPatchEmbedding(self.n_vars, configs.d_model, self.patch_len, configs.dropout)
        self.latest_attention = 0
        #.wavelet_embedding = WaveletEmbedding(d_channel=self.n_vars, swt=True, requires_grad=True, wv='db1', m=3, kernel_size=3)
        #self.inverse_wavelet_embedding = WaveletEmbedding(d_channel=self.n_vars, swt=False, requires_grad=True, wv='db1', m=3, kernel_size=3)
        self.PatchWavel
        etEmbedding = PatchWaveletEmbedding(self.n_vars, configs.d_model, self.patch_len, configs.dropout)
    
        self.encoder = Encoder(
            [
                EncoderLayer(
                    AttentionLayer(
                        FullAttention(False, configs.factor, attention_dropout=configs.dropout,
                                      output_attention=False, temp=configs.self_temp),
                        configs.d_model, configs.n_heads),
                    AttentionLayer(
                        FullAttention_temp(False, configs.factor, attention_dropout=configs.dropout,
                                      output_attention=True, temp=self.cross_temp), #temp 추가, FullAttention_temp 사용
                        configs.d_model, configs.n_heads),
                    configs.d_model,
                    configs.d_ff,
                    dropout=configs.dropout,
                    activation=configs.activation,
                    n_vars=self.n_vars # 코드 오류 수정 위해서 n_vars 추가
                )
                for l in range(configs.e_layers)
            ],
            norm_layer=torch.nn.LayerNorm(configs.d_model)
        )
        
        #self.head = PerVarProjectionHead(configs.enc_in, configs.d_model, configs.pred_len, dropout=configs.dropout, features=configs.features)

        # Use Single head
        self.head_nf = configs.d_model//(self.patch_num) #* (self.patch_num + 1)
        self.head = SingleProjectionHead(configs.enc_in, self.head_nf, configs.pred_len, head_dropout=configs.dropout)

    def forecast(self, x_enc, x_mark_enc, x_dec, x_mark_dec):
        if self.use_norm:
            # Normalization from Non-stationary Transformer
            means = x_enc.mean(1, keepdim=True).detach()
            x_enc = x_enc - means
            stdev = torch.sqrt(torch.var(x_enc, dim=1, keepdim=True, unbiased=False) + 1e-5)
            x_enc /= stdev

        _, _, N = x_enc.shape

        global_patch_embed, n_vars = self.global_patch_embedding(x_enc[:, :, -1].unsqueeze(-1).permute(0, 2, 1))

        enc_out, attn = self.encoder(global_patch_embed, n_vars)
        enc_out = torch.reshape(
            enc_out, (-1, n_vars, enc_out.shape[-2], enc_out.shape[-1]))
        # z: [bs x nvars x d_model x patch_num]
        enc_out = enc_out.permute(0, 1, 3, 2)

        dec_out = self.head(enc_out)  # z: [bs x nvars x target_window]
        #dec_out = dec_out.permute(0, 2, 1)

        if self.use_norm:
            # De-Normalization from Non-stationary Transformer
            dec_out = dec_out * (stdev[:, 0, -1:].unsqueeze(1).repeat(1, self.pred_len, 1))
            dec_out = dec_out + (means[:, 0, -1:].unsqueeze(1).repeat(1, self.pred_len, 1))

        return dec_out, attn


    def forecast_multi(self, x_enc, x_mark_enc, x_dec, x_mark_dec):
        if self.use_norm:
            # Normalization from Non-stationary Transformer
            means = x_enc.mean(1, keepdim=True).detach()
            x_enc = x_enc - means
            stdev = torch.sqrt(torch.var(x_enc, dim=1, keepdim=True, unbiased=False) + 1e-5)
            x_enc /= stdev

        B, _, N = x_enc.shape

        # Wavelet Embedding
        x_enc, n_vars = self.PatchWaveletEmbedding(x_enc.permute(0,2,1)) # [B, N, m+1, L]
        # for attention map 쓰는 모델
        enc_out, attn = self.encoder(x_enc) # [B*N, M, P, D]
        self.latest_attention = attn
        # for attention map 안 쓰는 모델
        # enc_out = self.encoder(global_patch_embed)
        glb_out = enc_out[:, :, -1, :]  # [B*N, M, D]
        glb_out = glb_out.reshape(B, N, -1, glb_out.shape[-1])  # [B, N, M, D]

        glb_out = self.inverse_wavelet_embedding(glb_out)  # [B, N, D]

        dec_out = self.head(glb_out)  # z: [bs x nvars x target_window]
        #dec_out = dec_out.permute(0, 2, 1)

        if self.use_norm:
            # De-Normalization from Non-stationary Transformer
            dec_out = dec_out * (stdev[:, 0, :].unsqueeze(1).repeat(1, self.pred_len, 1))
            dec_out = dec_out + (means[:, 0, :].unsqueeze(1).repeat(1, self.pred_len, 1))
        
        # return dec_out
        return dec_out, attn

    def forward(self, x_enc, x_mark_enc, x_dec, x_mark_dec, mask=None):
        if self.task_name == 'long_term_forecast' or self.task_name == 'short_term_forecast':
            if self.features == 'M':
                dec_out, attn = self.forecast_multi(x_enc, x_mark_enc, x_dec, x_mark_dec)
                return dec_out[:, -self.pred_len:, :]#, attn  # [B, L, D]
                # for attention map 안 쓰는 모델
                # dec_out = self.forecast_multi(x_enc, x_mark_enc, x_dec, x_mark_dec)
                # return dec_out[:, -self.pred_len:, :]  # [B, L, D]
            else:
                # for attention map
                dec_out = self.forecast(x_enc, x_mark_enc, x_dec, x_mark_dec)
                return dec_out[:, -self.pred_len:, :]  # [B, L, D]
                # for not attention map
                # dec_out = self.forecast(x_enc, x_mark_enc, x_dec, x_mark_dec)
                # return dec_out[:, -self.pred_len:, :], attn  # [B, L, D]
                
        else:
            return None