import torch
import torch.nn as nn
import torch.nn.functional as F

import sys
sys.path.append("models/")

from torch_geometric.nn import global_add_pool

from torch.nn.utils.rnn import pad_sequence
from mlp import MLP
from GPS import GPSModel



class GIN(nn.Module):
    def __init__(self, num_layers, num_mlp_layers, input_dim, hidden_dim, final_dropout, learn_eps, neighbor_pooling_type, device):
        '''
            num_layers: number of layers in the neural networks (INCLUDING the input layer)
            num_mlp_layers: number of layers in mlps (EXCLUDING the input layer)
            input_dim: dimensionality of input features
            hidden_dim: dimensionality of hidden units at ALL layers
            output_dim: number of classes for prediction
            final_dropout: dropout ratio on the final linear layer
            learn_eps: If True, learn epsilon to distinguish center nodes from neighboring nodes. If False, aggregate neighbors and center nodes altogether. 
            neighbor_pooling_type: how to aggregate neighbors (mean, average, or max)
            device: which device to use
        '''

        super(GIN, self).__init__()

        self.final_dropout = final_dropout
        self.device = device
        self.num_layers = num_layers
        self.neighbor_pooling_type = neighbor_pooling_type
        self.learn_eps = learn_eps
        self.eps = nn.Parameter(torch.zeros(self.num_layers-1))
        self.hidden_channels = hidden_dim

        ###List of MLPs
        self.mlps = torch.nn.ModuleList()

        ###List of batchnorms applied to the output of MLP (input of the final prediction linear layer)
        self.batch_norms = torch.nn.ModuleList()

        for layer in range(self.num_layers-1):
            if layer == 0:
                self.mlps.append(MLP(num_mlp_layers, input_dim, hidden_dim, hidden_dim))
            else:
                self.mlps.append(MLP(num_mlp_layers, hidden_dim, hidden_dim, hidden_dim))

            self.batch_norms.append(nn.BatchNorm1d(hidden_dim))



    def __preprocess_neighbors_maxpool(self, batch_graph):
        ###create padded_neighbor_list in concatenated graph

        #compute the maximum number of neighbors within the graphs in the current minibatch
        max_deg = max([graph.max_neighbor for graph in batch_graph])

        padded_neighbor_list = []
        start_idx = [0]


        for i, graph in enumerate(batch_graph):
            start_idx.append(start_idx[i] + len(graph.g))
            padded_neighbors = []
            for j in range(len(graph.neighbors)):
                #add off-set values to the neighbor indices
                pad = [n + start_idx[i] for n in graph.neighbors[j]]
                #padding, dummy data is assumed to be stored in -1
                pad.extend([-1]*(max_deg - len(pad)))

                #Add center nodes in the maxpooling if learn_eps is False, i.e., aggregate center nodes and neighbor nodes altogether.
                if not self.learn_eps:
                    pad.append(j + start_idx[i])

                padded_neighbors.append(pad)
            padded_neighbor_list.extend(padded_neighbors)

        return torch.LongTensor(padded_neighbor_list)


    def __preprocess_neighbors_sumavepool(self, batch_graph):
        ###create block diagonal sparse matrix

        edge_mat_list = []
        start_idx = [0]
        for i, graph in enumerate(batch_graph):
            start_idx.append(start_idx[i] + len(graph.g))
            edge_mat_list.append(graph.edge_mat + start_idx[i])
        Adj_block_idx = torch.cat(edge_mat_list, 1)
        Adj_block_elem = torch.ones(Adj_block_idx.shape[1])

        #Add self-loops in the adjacency matrix if learn_eps is False, i.e., aggregate center nodes and neighbor nodes altogether.

        if not self.learn_eps:
            num_node = start_idx[-1]
            self_loop_edge = torch.LongTensor([range(num_node), range(num_node)])
            elem = torch.ones(num_node)
            Adj_block_idx = torch.cat([Adj_block_idx, self_loop_edge], 1)
            Adj_block_elem = torch.cat([Adj_block_elem, elem], 0)

        Adj_block = torch.sparse.FloatTensor(Adj_block_idx, Adj_block_elem, torch.Size([start_idx[-1],start_idx[-1]]))

        return Adj_block.to(self.device)



    def maxpool(self, h, padded_neighbor_list):
        ###Element-wise minimum will never affect max-pooling

        dummy = torch.min(h, dim = 0)[0]
        h_with_dummy = torch.cat([h, dummy.reshape((1, -1)).to(self.device)])
        pooled_rep = torch.max(h_with_dummy[padded_neighbor_list], dim = 1)[0]
        return pooled_rep


    def next_layer_eps(self, h, layer, padded_neighbor_list = None, Adj_block = None):
        ###pooling neighboring nodes and center nodes separately by epsilon reweighting. 

        if self.neighbor_pooling_type == "max":
            ##If max pooling
            pooled = self.maxpool(h, padded_neighbor_list)
        else:
            #If sum or average pooling

            pooled = torch.spmm(Adj_block, h)
            if self.neighbor_pooling_type == "average":
                #If average pooling
                degree = torch.spmm(Adj_block, torch.ones((Adj_block.shape[0], 1)).to(self.device))
                pooled = pooled/degree

        #Reweights the center node representation when aggregating it with its neighbors
        pooled = pooled + (1 + self.eps[layer])*h
        pooled_rep = self.mlps[layer](pooled)
        h = self.batch_norms[layer](pooled_rep)

        #non-linearity
        h = F.relu(h)
        return h


    def next_layer(self, h, layer, padded_neighbor_list = None, Adj_block = None):
        ###pooling neighboring nodes and center nodes altogether  
            
        if self.neighbor_pooling_type == "max":
            ##If max pooling
            pooled = self.maxpool(h, padded_neighbor_list)
        else:
            #If sum or average pooling
            pooled = torch.spmm(Adj_block, h)
            if self.neighbor_pooling_type == "average":
                #If average pooling
                degree = torch.spmm(Adj_block, torch.ones((Adj_block.shape[0], 1)).to(self.device))
                pooled = pooled/degree

        #representation of neighboring and center nodes 
        pooled_rep = self.mlps[layer](pooled)

        h = self.batch_norms[layer](pooled_rep)

        #non-linearity
        h = F.relu(h)
        return h

    def forward(self, data):
        adj = data.adj
        batch_size = data.batch[-1] + 1
        X_concat = data.x.float()
        # graph_pool = self.__preprocess_graphpool(batch_graph)
# 
        if self.neighbor_pooling_type == "max":
            padded_neighbor_list = self.__preprocess_neighbors_maxpool(batch_graph)
        else:
            Adj_block = adj

        #list of hidden representation at each layer (including input)
        hidden_rep = [X_concat]
        h = X_concat

        for layer in range(self.num_layers-1):
            h = h.float()
            if self.neighbor_pooling_type == "max" and self.learn_eps:
                h = self.next_layer_eps(h, layer, padded_neighbor_list = padded_neighbor_list)
            elif not self.neighbor_pooling_type == "max" and self.learn_eps:
                h = self.next_layer_eps(h, layer, Adj_block = Adj_block)
            elif self.neighbor_pooling_type == "max" and not self.learn_eps:
                h = self.next_layer(h, layer, padded_neighbor_list = padded_neighbor_list)
            elif not self.neighbor_pooling_type == "max" and not self.learn_eps:
                h = self.next_layer(h, layer, Adj_block = Adj_block)
        return h





class GlobalGIN(nn.Module):
    def __init__(self, num_layers, num_mlp_layers, final_mlp_layers, input_dim, hidden_dim, output_dim, evec_len, final_dropout, learn_eps, neighbor_pooling_type, device, alt_node_target_sizes={}, alt_graph_target_sizes={}, model_name=None, head_type = "concat"):
        '''
            num_layers: number of layers in the neural networks (INCLUDING the input layer)
            num_mlp_layers: number of layers in mlps (EXCLUDING the input layer)
            input_dim: dimensionality of input features
            hidden_dim: dimensionality of hidden units at ALL layers
            output_dim: number of classes for prediction
            final_dropout: dropout ratio on the final linear layer
            learn_eps: If True, learn epsilon to distinguish center nodes from neighboring nodes. If False, aggregate neighbors and center nodes altogether. 
            neighbor_pooling_type: how to aggregate neighbors (mean, average, or max)
            device: which device to use
        '''

        super(GlobalGIN, self).__init__()
        self.model_name = model_name
        if model_name == "GPS":
            print("USING GPS")
            self.GIN = GPSModel(num_layers, num_mlp_layers, input_dim, hidden_dim, final_dropout, learn_eps, neighbor_pooling_type, device)
        else:
            print("Using basic GIN")
            self.GIN = GIN(num_layers, num_mlp_layers, input_dim, hidden_dim, final_dropout, learn_eps, neighbor_pooling_type, device)
        self.final_dropout = final_dropout
        self.device = device
        self.num_layers = num_layers
        self.neighbor_pooling_type = neighbor_pooling_type
        self.learn_eps = learn_eps
        self.eps = nn.Parameter(torch.zeros(self.num_layers-1))
        self.head_type = head_type


        self.evec_len = evec_len

        if head_type == "concat":
            print("USING CONCAT MLP")
            self.final_mlp = MLP(final_mlp_layers, evec_len * hidden_dim, evec_len * hidden_dim, evec_len * output_dim).to(device) # final layer, which processes concatenated node embeddings and outputs full eigenvector matrix
        elif head_type == "per_node":
            print("USING PER-NODE MLP")
            self.final_mlp = MLP(final_mlp_layers, hidden_dim,  hidden_dim, output_dim).to(device) # final layer, which processes concatenated node embeddings and outputs full eigenvector matrix

        self.alt_targets = False

        self.alt_node_targets_heads = {}
        if len(alt_node_target_sizes) > 0:
            self.alt_targets = True
            for key in alt_node_target_sizes.keys():
                size = alt_node_target_sizes[key]

                if head_type == "concat":
                    self.alt_node_targets_heads[key] = MLP(final_mlp_layers, evec_len * hidden_dim, evec_len * hidden_dim, evec_len * size).to(device) # final layer, which processes concatenated node embeddings and outputs full eigenvector matrix
                elif head_type == "per_node":
                    self.alt_node_targets_heads[key] = MLP(final_mlp_layers, hidden_dim,  hidden_dim, size).to(device) # final layer, which processes concatenated node embeddings and outputs full eigenvector matrix

            


        self.alt_graph_targets_heads = {}
        if len(alt_graph_target_sizes) > 0:
            self.alt_targets = True
            for key in alt_graph_target_sizes.keys():
                size = alt_graph_target_sizes[key]

                if head_type == "concat":
                    self.alt_graph_targets_heads[key] = MLP(final_mlp_layers, evec_len * hidden_dim, evec_len * hidden_dim, evec_len * size).to(device) # final layer, which processes concatenated node embeddings and outputs full eigenvector matrix
                elif head_type == "per_node":
                    self.alt_graph_targets_heads[key] = MLP(final_mlp_layers, hidden_dim,  hidden_dim, size).to(device) # final layer, which processes concatenated node embeddings and outputs full eigenvector matrix



    def forward(self, data):
        data.x = data.x.float()
        batch = data.batch
        batch_size = batch[-1] + 1

        h = self.GIN(data)

        final_out = self.head_forward(self.final_mlp, h, batch, batch_size)


        if self.alt_targets:
            alt_targets_out = {}
            
            for key in self.alt_node_targets_heads.keys():
                target_mlp = self.alt_node_targets_heads[key]
                alt_targets_out[key] = self.head_forward(target_mlp, h, batch, batch_size)
            
            # pooled_h = global_add_pool(h, batch) 
            for key in self.alt_graph_targets_heads.keys():
                target_mlp = self.alt_graph_targets_heads[key]
                
                alt_targets_out[key] = self.head_forward(target_mlp, h, batch, batch_size, True)
                # no concat operation needed for graph-level prediction, features already pooled into a global feature 
            return final_out, alt_targets_out   

        else:
        
            return final_out


    def head_forward(self, mlp, h, batch, batch_size):
        if self.head_type == "concat":
            out = self.eigvec_forward(mlp, h, batch, batch_size)
        elif self.head_type == "per_node":
            out = self.final_mlp(h)
        return out


    def eigvec_forward(self, mlp, h, batch, batch_size, graph_level_out = False):

        h_batches = [h[batch == i] for i in range(batch_size)]
        padded_h = torch.zeros(0, h.shape[-1]).to(self.device)
        pad_tracker = torch.ones(0).to(self.device) # 1s at original indices, 0 at other-- this is to "unpad" later 

        for h_batch in h_batches:
            pad_tracker = torch.cat((pad_tracker, F.pad(torch.ones(h_batch.shape[0]).to(self.device), (0, self.evec_len - h_batch.shape[0]), value=0)), dim=0)

            padded_h = torch.cat((padded_h, F.pad(h_batch, (0,0,0, self.evec_len - h_batch.shape[0]), value=0)), dim=0)

        # padded_h = pad_sequence(graphs, batch_first=True) # pad so there are effectively evec_len nodes 
        flattened_h = padded_h.view(batch_size, self.evec_len * h.shape[-1]) # (B, N*H) where N is the max node count, i.e. evec_len
        
        if graph_level_out:
            return mlp(flattened_h)
            
        padded_out = mlp(flattened_h).view(batch_size * self.evec_len, -1) # 

        final_out = padded_out[pad_tracker.bool()] # FILTER BY PAD_TRACKER HERE
        return final_out
