"""
This script is used to evaluate QSP on HHL
Currently, this example is modified directly based on qiskit v0.37:
    https://docs.quantum.ibm.com/api/qiskit/0.37/qiskit.algorithms.linear_solvers.HHL
However, some APIs have already deprecated, so that we need to evaluate HHL in another isolate
environment.

In the future, we will implement HHL in newer version of qiskit to make it compatible with the
main environment used by the SuperEncoder repository

"""

## qiskit               0.37.0
## qiskit-aer           0.10.4
## qiskit-ibmq-provider 0.19.2
## qiskit-terra         0.21.0

import os
import sys

import numpy as np
from qiskit import QuantumCircuit
from qiskit.algorithms.linear_solvers.hhl import HHL
from qiskit.algorithms.linear_solvers.matrices import TridiagonalToeplitz
from qiskit.algorithms.linear_solvers.observables import MatrixFunctional

if len(sys.argv) != 2:
    print(f"Usage: python ${sys.argv[0]} <path-to-target-state-plain-text>")
    sys.exit(1)

state_file_path = sys.argv[1]
idx = state_file_path.rfind("_")
file_path_prefix = state_file_path[:idx]


def get_rhs(file_path):
    with open(file_path, "r") as f:
        print(f"loading {file_path}")
        # FIXME: we should avoid using `eval`
        right_hand_side = np.array(eval(f.read()))
        return right_hand_side / np.linalg.norm(right_hand_side)


rhs_orig = get_rhs(state_file_path)
rhs_superencoder = get_rhs(file_path_prefix + "_superencoder.txt")
rhs_aae = get_rhs(file_path_prefix + "_aae.txt")


matrix = TridiagonalToeplitz(
    4, 1, 1 / 3, trotter_steps=2
)  # num_state_qubits is fixed at 4
observable = MatrixFunctional(1, 1 / 2)

# Initial state circuit
num_qubits = matrix.num_state_qubits


def run_hhl(rhs):
    qc_original = QuantumCircuit(num_qubits)
    qc_original.isometry(rhs, list(range(num_qubits)), None)
    hhl = HHL()
    solution = hhl.solve(matrix, qc_original, observable)
    approx_res = solution.observable
    print(f"result: {approx_res}")


run_hhl(rhs_orig)
run_hhl(rhs_superencoder)
run_hhl(rhs_aae)
