import math
import torch
import torch.nn as nn
import torch.nn.init as init
from spikingjelly.clock_driven import functional, surrogate, layer, neuron

__all__ = [
    'VGG', 'snnvgg11', 'snnvgg11_bn', 'snnvgg13', 'snnvgg13_bn', 'snnvgg16', 'snnvgg16_bn',
    'snnvgg19_bn', 'snnvgg19',
]


class VGG(nn.Module):
    '''
    VGG model
    '''

    def __init__(self, features):
        super(VGG, self).__init__()
        self.sn1 = neuron.MultiStepIFNode(detach_reset=True)
        self.sn2 = neuron.MultiStepIFNode(detach_reset=True)

        self.features = features
        self.classifier1 = layer.SeqToANNContainer(
            nn.Linear(512, 512, bias=False),
            nn.BatchNorm1d(512))

        self.classifier2 = layer.SeqToANNContainer(
            nn.Linear(512, 512, bias=False),
            nn.BatchNorm1d(512))
        self.fc = nn.Linear(512, 100)

        



        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                n = m.kernel_size[0] * m.kernel_size[1] * m.out_channels
                m.weight.data.normal_(0, math.sqrt(2. / n))
            elif isinstance(m, nn.BatchNorm2d):
                m.weight.data.fill_(1)
                m.bias.data.zero_()
            elif isinstance(m, nn.Linear):
                m.weight.data.normal_(0, 0.02)

    def forward(self, x):

        x.unsqueeze_(0)
        out = x.repeat(4, 1, 1, 1, 1)

        # print(out.shape)
        out = self.features(out)
        out = torch.flatten(out, 2)
        out=self.classifier1(out)
        out=self.sn1(out)
        out = self.classifier2(out)
        out = self.sn2(out)




        return self.fc(out.mean(dim=0))





def make_layers(cfg, batch_norm=False):
    layers = []
    in_channels = 3
    for v in cfg:
        if v == 'M':
            layers += [layer.SeqToANNContainer(nn.MaxPool2d(kernel_size=2, stride=2))]
        else:
            conv2d = layer.SeqToANNContainer(nn.Conv2d(in_channels, v, kernel_size=3, padding=1, bias=False))
            if batch_norm:
                layers += [conv2d, layer.SeqToANNContainer(nn.BatchNorm2d(v)),
                           neuron.MultiStepIFNode(detach_reset=True)]
            else:
                layers += [conv2d, neuron.MultiStepIFNode(detach_reset=True)]

            in_channels = v
    return nn.Sequential(*layers)


cfg = {
    'A': [64, 'M', 128, 'M', 256, 256, 'M', 512, 512, 'M', 512, 512, 'M'],
    'B': [64, 64, 'M', 128, 128, 'M', 256, 256, 'M', 512, 512, 'M', 512, 512, 'M'],
    'D': [64, 64, 'M', 128, 128, 'M', 256, 256, 256, 'M', 512, 512, 512, 'M', 512, 512, 512, 'M'],
    'E': [64, 64, 'M', 128, 128, 'M', 256, 256, 256, 256, 'M', 512, 512, 512, 512, 'M',
          512, 512, 512, 512, 'M'],
}


def snnvgg11():
    """VGG 11-layer model (configuration "A")"""
    return VGG(make_layers(cfg['A']))


def snnvgg11_bn():
    """VGG 11-layer model (configuration "A") with batch normalization"""
    return VGG(make_layers(cfg['A'], batch_norm=True))


def snnvgg13():
    """VGG 13-layer model (configuration "B")"""
    return VGG(make_layers(cfg['B']))


def snnvgg13_bn():
    """VGG 13-layer model (configuration "B") with batch normalization"""
    return VGG(make_layers(cfg['B'], batch_norm=True))


def snnvgg16():
    """VGG 16-layer model (configuration "D")"""
    return VGG(make_layers(cfg['D']))


def snnvgg16_bn():
    """VGG 16-layer model (configuration "D") with batch normalization"""
    return VGG(make_layers(cfg['D'], batch_norm=True))


def snnvgg19():
    """VGG 19-layer model (configuration "E")"""
    return VGG(make_layers(cfg['E']))


def snnvgg19_bn():
    """VGG 19-layer model (configuration 'E') with batch normalization"""
    return VGG(make_layers(cfg['E'], batch_norm=True))
