import math
import torch
import torch.nn as nn
import torch.nn.functional as F
from spikingjelly.clock_driven import layer, neuron



class BasicBlock(nn.Module):
    def __init__(self, in_planes, out_planes, stride, dropRate=0.0):
        super(BasicBlock, self).__init__()
        self.bn1 = layer.SeqToANNContainer(nn.BatchNorm2d(in_planes))
        self.relu1 = neuron.MultiStepIFNode(detach_reset=True)
        self.conv1 = layer.SeqToANNContainer(nn.Conv2d(in_planes, out_planes, kernel_size=3, stride=stride,
                               padding=1, bias=False))
        self.bn2 = layer.SeqToANNContainer(nn.BatchNorm2d(out_planes))
        self.relu2 = neuron.MultiStepIFNode(detach_reset=True)
        self.conv2 = layer.SeqToANNContainer(nn.Conv2d(out_planes, out_planes, kernel_size=3, stride=1,
                               padding=1, bias=False))
        self.droprate = dropRate
        self.equalInOut = (in_planes == out_planes)
        self.convShortcut = (not self.equalInOut) and layer.SeqToANNContainer(nn.Conv2d(in_planes, out_planes, kernel_size=1, stride=stride,
                               padding=0, bias=False)) or None

    def forward(self, x):
        if not self.equalInOut:
            x = self.relu1(self.bn1(x))
        else:
            out = self.relu1(self.bn1(x))
        out = self.relu2(self.bn2(self.conv1(out if self.equalInOut else x)))
        if self.droprate > 0:
            out = F.dropout(out, p=self.droprate, training=self.training)
        out = self.conv2(out)
        return torch.add(x if self.equalInOut else self.convShortcut(x), out)

class NetworkBlock(nn.Module):
    def __init__(self, nb_layers, in_planes, out_planes, block, stride, dropRate=0.0):
        super(NetworkBlock, self).__init__()
        self.layer = self._make_layer(block, in_planes, out_planes, nb_layers, stride, dropRate)

    def _make_layer(self, block, in_planes, out_planes, nb_layers, stride, dropRate):
        layers = []
        for i in range(int(nb_layers)):
            layers.append(block(i == 0 and in_planes or out_planes, out_planes, i == 0 and stride or 1, dropRate))
        return nn.Sequential(*layers)

    def forward(self, x):
        return self.layer(x)

class SNNWideResNet(nn.Module):
    def __init__(self, depth=16, num_classes=100, widen_factor=4, dropRate=0.0):
        super(SNNWideResNet, self).__init__()
        nChannels = [16, 16*widen_factor, 32*widen_factor, 64*widen_factor]
        assert((depth - 4) % 6 == 0)
        n = (depth - 4) / 6
        block = BasicBlock

        self.conv1 = nn.Conv2d(3, nChannels[0], kernel_size=3, stride=1, padding=1, bias=False)

        self.block1 = NetworkBlock(n, nChannels[0], nChannels[1], block, 1, dropRate)

        self.block2 = NetworkBlock(n, nChannels[1], nChannels[2], block, 2, dropRate)

        self.block3 = NetworkBlock(n, nChannels[2], nChannels[3], block, 2, dropRate)

        self.bn1 = layer.SeqToANNContainer( nn.BatchNorm2d(nChannels[3]))
        self.relu = neuron.MultiStepIFNode(detach_reset=True)
        self.fc = nn.Linear(nChannels[3], num_classes)
        self.nChannels = nChannels
        self.avgpool = layer.SeqToANNContainer(nn.AvgPool2d(8))
        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                n = m.kernel_size[0] * m.kernel_size[1] * m.out_channels
                m.weight.data.normal_(0, math.sqrt(2. / n))
            elif isinstance(m, nn.BatchNorm2d):
                m.weight.data.fill_(1)
                m.bias.data.zero_()
            elif isinstance(m, nn.Linear):
                m.bias.data.zero_()

    def forward(self, x):
        out = self.conv1(x)
        out.unsqueeze_(0)
        out = out.repeat(4, 1, 1, 1, 1)
        out = self.block1(out)
        out = self.block2(out)
        out = self.block3(out)
        out = self.relu(self.bn1(out))
        out = self.avgpool(out)
        out = torch.flatten(out, 2)
        return self.fc(out.mean(dim=0))

