import argparse
import os
import random
import shutil
import time
import warnings
from enum import Enum

import torch
import torch.backends.cudnn as cudnn
import torch.nn as nn

import random
import os 
import numpy as np 
import PIL
from PIL import Image
import pdb 

import torch
from torch import autocast
from diffusers import StableDiffusionPipeline
from diffusers import DPMSolverMultistepScheduler

import torchvision.datasets as datasets
import torchvision.transforms as transforms
from imagenet_labels import ind2name

import warnings
warnings.filterwarnings("ignore")

parser = argparse.ArgumentParser(description='PyTorch Diffusion based data generator')
parser.add_argument('data', metavar='DIR', nargs='?', default='imagenet',
                    help='path to dataset (default: imagenet)')

parser.add_argument('--start_seed', default=None, type=int,
                    help='seed for initializing training. ')
parser.add_argument('--end_seed', default=None, type=int,
                    help='seed for initializing training. ')

parser.add_argument('--method', type=str, default='none',
                    help='method for data generation (none, positive, negative, prompt)')
parser.add_argument('--noise', default=0.5, type=float, help='noise rate for forward diffusion')
parser.add_argument('--nway', default=10, type=int, help='n-way classification')
parser.add_argument('--budget', default=100, type=int, help='max generated samples per class')
parser.add_argument('--kshot', default=5, type=int, help='n-shot classification')
parser.add_argument('--output_dir', type=str, default='./data/',
                    help='path to store the generated data')
parser.add_argument('--confusion_matrix', type=str, default='./data/',
                    help='path to confusion matrix')
parser.add_argument('--prompt', type=str, default='',
                    help='path to confusion matrix')

parser.add_argument('--imagenet', action='store_true', help="generate from imagenet")

parser.add_argument('--world-size', default=-1, type=int,
                    help='number of nodes for distributed training')
parser.add_argument('--rank', default=-1, type=int,
                    help='node rank for distributed training')
parser.add_argument('--dist-url', default='tcp://224.66.41.62:23456', type=str,
                    help='url used to set up distributed training')
parser.add_argument('--dist-backend', default='nccl', type=str,
                    help='distributed backend')

parser.add_argument('--gpu', default=None, type=int,
                    help='GPU id to use.')
parser.add_argument('--multiprocessing-distributed', action='store_true',
                    help='Use multi-processing distributed training to launch '
                         'N processes per node, which has N GPUs. This is the '
                         'fastest way to use PyTorch for either single node or '
                         'multi node data parallel training')
parser.add_argument('--dummy', action='store_true', help="use fake data to benchmark")

best_acc1 = 0



def set_all_seeds(seed):
    random.seed(seed)
    np.random.seed(seed)
    torch.manual_seed(seed)
    torch.cuda.manual_seed(seed)
    torch.backends.cudnn.deterministic = True

def main():
    args = parser.parse_args()

    cudnn.deterministic = True
    cudnn.benchmark = False

    if args.gpu is not None:
        warnings.warn('You have chosen a specific GPU. This will completely '
                      'disable data parallelism.')

    if args.dist_url == "env://" and args.world_size == -1:
        args.world_size = int(os.environ["WORLD_SIZE"])

    args.distributed = args.world_size > 1 or args.multiprocessing_distributed

    if torch.cuda.is_available():
        ngpus_per_node = torch.cuda.device_count()
    else:
        ngpus_per_node = 1
        
    main_worker(args.gpu, ngpus_per_node, args)


def main_worker(gpu, ngpus_per_node, args):
    
    pipe = StableDiffusionPipeline.from_pretrained("runwayml/stable-diffusion-v1-5", torch_dtype=torch.float16)
    pipe = pipe.to('cuda')
    pipe.scheduler = DPMSolverMultistepScheduler.from_config(pipe.scheduler.config)
    type(pipe).__call__ = __call__
    
    NUM_INFERENCE_STEPS = 50 if isinstance(pipe.scheduler, DPMSolverMultistepScheduler) else 100
    total_seeds = args.end_seed - args.start_seed 
    
    for sind, seed in enumerate(range(args.start_seed, args.end_seed)):
        
        print("processing seed {}, {}/{}".format(seed, sind+1, total_seeds))

        set_all_seeds(seed)
        
        dataset_path = args.output_dir
        dataset_path = os.path.join(dataset_path , 'dataset{}'.format(seed))
        src_data = os.path.join(args.data , 'dataset{}/train/'.format(seed))
        
        dataset = datasets.ImageFolder(
            src_data,
            transforms.Compose([
                transforms.Resize(512),
                transforms.CenterCrop(512),
            ]))
    
        cls_path = [f.path for f in os.scandir(src_data) if f.is_dir()]
        cls_path = [c.split("/")[-1] for c in cls_path]
        cls_path = np.sort(cls_path)
        cls_names = [c[10:] for c in cls_path]

        if 'txt2img' in args.method: 
            train_path = os.path.join(dataset_path, 'train_{}'.format(args.method))
        else:
            train_path = os.path.join(dataset_path, 'train_{}_noise{}'.format(args.method, args.noise))
        os.makedirs(train_path, exist_ok = True) 

        t = transforms.Compose([transforms.Resize(512),transforms.CenterCrop(512),])


        for ind, cls in enumerate(cls_names): 

            class_path = os.path.join(train_path, cls_path[ind])
            os.makedirs(class_path, exist_ok = True)

            if 'genie' in args.method: 
                imgs = [c[0] for c in dataset.samples if c[1]!=ind]
            else:
                imgs = [c[0] for c in dataset.samples if c[1]==ind]

            print("processing class {}, {}/{}".format(cls, ind+1 , len(cls_names)))

            for k,img in enumerate(imgs): 

                if 'genie' in args.method: 
                    prompt = 'a photo of a {}.'.format(cls)

                    img_ = Image.open(img).convert('RGB')
                    img_ = t(img_)

                    clean_z = encode_latents(img_, pipe)
                    noisy_z = diffusion_forward(args.noise, clean_z, pipe)
                    output  = diffusion_reverse(prompt, args.noise, noisy_z, pipe, 
                                                num_inference_steps=NUM_INFERENCE_STEPS, output_type='pil').images[0]
                    img_path = os.path.join(class_path, '{}_genie.jpg'.format(k))
                    output.save(img_path)

                if 'img2img' in args.method:

                    prompt = 'a photo of a {}.'.format(cls)
                    img_ = Image.open(img).convert('RGB')
                    img_ = t(img_)

                    for j in range(len(cls_names)-1):
                        clean_z = encode_latents(img_, pipe)
                        noisy_z = diffusion_forward(args.noise, clean_z, pipe)
                        output  = diffusion_reverse(prompt, args.noise, noisy_z, pipe,
                                                    num_inference_steps=NUM_INFERENCE_STEPS, output_type='pil').images[0]
                        img_path = os.path.join(class_path, '{}img2img{}.jpg'.format(k,j))
                        output.save(img_path)

                if 'txt2img' in args.method:
                    prompt = 'a photo of a {}.'.format(cls)
                    img_ = Image.open(img).convert('RGB')
                    img_ = t(img_)

                    for j in range(len(cls_names)-1):
                        clean_z = encode_latents(img_, pipe) 
                        clean_z = torch.zeros_like(clean_z)
                        noisy_z = diffusion_forward(0.999, clean_z, pipe)
                        output  = diffusion_reverse(prompt, 0.999, noisy_z, pipe,
                                                    num_inference_steps=NUM_INFERENCE_STEPS, output_type='pil').images[0]
                        img_path = os.path.join(class_path, '{}_txt2img{}.jpg'.format(k,j))
                        output.save(img_path)




import torch.nn.functional as F
from torchvision import transforms as T

def encode_latents(img, pipe):
    # Convert image to float and preprocess it, same as in
    # huggingface/diffusers/blob/main/examples/unconditional_image_generation/train_unconditional.py
    transform = T.Compose([T.PILToTensor(),
                        T.ConvertImageDtype(torch.float),
                        T.Resize(512),
                        T.CenterCrop(512),
                        T.Normalize([0.5], [0.5])])


    with torch.no_grad():
        tensor = transform(img).half().to(pipe.device)
        tensor = torch.unsqueeze(tensor, 0)

    # Project image into the latent space
    clean_z = pipe.vae.encode(tensor).latent_dist.mode() # From huggingface/diffusers/blob/main/src/diffusers/models/vae.py
    clean_z = 0.18215 * clean_z

    return clean_z


def diffusion_forward(percent_noise, latents, pipe):
    '''
    Add noise to the latents via the pipe noise scheduler, according to percent_noise.
    '''
    assert percent_noise <= 0.999
    assert percent_noise >= 0.02

    # Add noise to the latent variable
    # (this is the forward diffusion process)
    noise = torch.randn(latents.shape).to(pipe.device)
    timestep = torch.Tensor([int(pipe.scheduler.config.num_train_timesteps * percent_noise)]).to(pipe.device).long()
    z = pipe.scheduler.add_noise(latents, noise, timestep).half()

    return z


def diffusion_reverse(prompt, percent_noise, latents, pipe, num_inference_steps=100, output_type='pil', guidance_scale=7.5):
    '''
    Denoise the noisy latents according to percent_noise.
    '''
    assert percent_noise <= 0.999
    assert percent_noise >= 0.02

    with autocast('cuda'):
        return pipe(prompt=prompt, percent_noise=percent_noise, latents=latents,
                    num_inference_steps=num_inference_steps, output_type=output_type)    
    
    
from typing import Any, Callable, Dict, List, Optional, Union
from diffusers.pipelines.stable_diffusion import StableDiffusionPipelineOutput

@torch.no_grad()
def __call__(
    self,
    prompt: Union[str, List[str]],
    percent_noise: int,
    height: Optional[int] = None,
    width: Optional[int] = None,
    num_inference_steps: int = 50,
    guidance_scale: float = 7.5,
    negative_prompt: Optional[Union[str, List[str]]] = None,
    num_images_per_prompt: Optional[int] = 1,
    eta: float = 0.0,
    generator: Optional[Union[torch.Generator, List[torch.Generator]]] = None,
    latents: Optional[torch.FloatTensor] = None,
    prompt_embeds: Optional[torch.FloatTensor] = None,
    negative_prompt_embeds: Optional[torch.FloatTensor] = None,
    output_type: Optional[str] = "pil",
    return_dict: bool = True,
    callback: Optional[Callable[[int, int, torch.FloatTensor], None]] = None,
    callback_steps: int = 1,
    cross_attention_kwargs: Optional[Dict[str, Any]] = None,
):

    height = height or self.unet.config.sample_size * self.vae_scale_factor
    width = width or self.unet.config.sample_size * self.vae_scale_factor

    self.check_inputs(
        prompt, height, width, callback_steps, negative_prompt, prompt_embeds, negative_prompt_embeds
    )

    if prompt is not None and isinstance(prompt, str):
        batch_size = 1
    elif prompt is not None and isinstance(prompt, list):
        batch_size = len(prompt)
    else:
        batch_size = prompt_embeds.shape[0]

    device = self._execution_device
    do_classifier_free_guidance = guidance_scale > 1.0

    prompt_embeds = self._encode_prompt(
        prompt,
        device,
        num_images_per_prompt,
        do_classifier_free_guidance,
        negative_prompt,
        prompt_embeds=prompt_embeds,
        negative_prompt_embeds=negative_prompt_embeds,
    )

    self.scheduler.set_timesteps(num_inference_steps, device=device)
    timesteps = self.scheduler.timesteps

    num_channels_latents = self.unet.config.in_channels
    latents = self.prepare_latents(
        batch_size * num_images_per_prompt,
        num_channels_latents,
        height,
        width,
        prompt_embeds.dtype,
        device,
        generator,
        latents,
    )

    extra_step_kwargs = self.prepare_extra_step_kwargs(generator, eta)

    num_warmup_steps = len(timesteps) - num_inference_steps * self.scheduler.order
    with self.progress_bar(total=num_inference_steps) as progress_bar:
        for i, t in enumerate(timesteps):

            if t - 1 > 1000 * percent_noise:
                continue

            latent_model_input = torch.cat([latents] * 2) if do_classifier_free_guidance else latents
            latent_model_input = self.scheduler.scale_model_input(latent_model_input, t)

            noise_pred = self.unet(
                latent_model_input,
                t,
                encoder_hidden_states=prompt_embeds,
                cross_attention_kwargs=cross_attention_kwargs,
            ).sample

            if do_classifier_free_guidance:
                noise_pred_uncond, noise_pred_text = noise_pred.chunk(2)
                noise_pred = noise_pred_uncond + guidance_scale * (noise_pred_text - noise_pred_uncond)

            latents = self.scheduler.step(noise_pred, t, latents, **extra_step_kwargs).prev_sample

            if i == len(timesteps) - 1 or ((i + 1) > num_warmup_steps and (i + 1) % self.scheduler.order == 0):
                progress_bar.update()
                if callback is not None and i % callback_steps == 0:
                    callback(i, t, latents)

    if output_type == "latent":
        image = latents
        has_nsfw_concept = None
    elif output_type == "pil":
        image = self.decode_latents(latents)
        image = self.numpy_to_pil(image)
    else:
        image = self.decode_latents(latents)

    if hasattr(self, "final_offload_hook") and self.final_offload_hook is not None:
        self.final_offload_hook.offload()

    if not return_dict:
        return (image, has_nsfw_concept)

    return StableDiffusionPipelineOutput(images=image, nsfw_content_detected=False)


if __name__ == '__main__':
    main()