import torch
import torch.nn as nn
from timm.models.layers import DropPath
from spikingjelly.clock_driven.neuron import (
    MultiStepLIFNode,
    MultiStepParametricLIFNode,
)
from . import mlif

class Erode(nn.Module):
    def __init__(self) -> None:
        super().__init__()
        self.pool = nn.MaxPool3d(
            kernel_size=(1, 3, 3), stride=(1, 1, 1), padding=(0, 1, 1)
        )

    def forward(self, x):
        return self.pool(x)


class MS_MLP_Conv(nn.Module):
    def __init__(
        self,
        in_features,
        hidden_features=None,
        out_features=None,
        drop=0.0,
        spike_mode="lif",
        mlif_channels=1,
        scaled=False,
        layer=0,
    ):
        super().__init__()
        out_features = out_features or in_features
        hidden_features = hidden_features or in_features
        self.res = in_features == hidden_features
        self.fc1_conv = nn.Conv2d(in_features, hidden_features, kernel_size=1, stride=1)
        self.fc1_bn = nn.BatchNorm2d(hidden_features)
        if spike_mode == "lif":
            self.fc1_lif = MultiStepLIFNode(tau=2.0, detach_reset=True, backend="cupy")
        elif spike_mode == "mlif":
            self.fc1_lif = mlif.MultiStepMLIFNode(tau=2.0, detach_reset=True, channels=mlif_channels, backend="torch", scaled=scaled)
        elif spike_mode == "fast_mlif":
            self.fc1_lif = mlif.FastMultiStepMLIFNode(tau=2.0, detach_reset=True, channels=mlif_channels, backend="cupy", scaled=scaled)
        elif spike_mode == "plif":
            self.fc1_lif = MultiStepParametricLIFNode(
                init_tau=2.0, detach_reset=True, backend="cupy"
            )

        self.fc2_conv = nn.Conv2d(
            hidden_features, out_features, kernel_size=1, stride=1
        )
        self.fc2_bn = nn.BatchNorm2d(out_features)
        if spike_mode == "lif":
            self.fc2_lif = MultiStepLIFNode(tau=2.0, detach_reset=True, backend="cupy")
        elif spike_mode == "mlif":
            self.fc2_lif = mlif.MultiStepMLIFNode(tau=2.0, detach_reset=True, channels=mlif_channels, backend="torch", scaled=scaled)
        elif spike_mode == "fast_mlif":
            self.fc2_lif = mlif.FastMultiStepMLIFNode(tau=2.0, detach_reset=True, channels=mlif_channels, backend="cupy", scaled=scaled)
        elif spike_mode == "plif":
            self.fc2_lif = MultiStepParametricLIFNode(
                init_tau=2.0, detach_reset=True, backend="cupy"
            )

        self.c_hidden = hidden_features
        self.c_output = out_features
        self.layer = layer
        self.mlif_channels = mlif_channels

    def forward(self, x, hook=None):
        T, B, C, H, W = x.shape
        identity = x

        x = self.fc1_lif(x)
        if hook is not None:
            hook[self._get_name() + str(self.layer) + "_fc1_lif"] = {"tensor": x.detach()}
        x = self.fc1_conv(x.flatten(0, 1))
        if hook is not None:
            hook[self._get_name() + str(self.layer) + "_fc1_lif"]["flops"] = torch.prod(torch.tensor(list(self.fc1_conv.weight.shape))) * x.shape[-1] * x.shape[-2]
        x = self.fc1_bn(x).reshape(T, B, self.c_hidden, H, W).contiguous()
        if self.res:
            x = identity + x
            identity = x
        x = self.fc2_lif(x)
        if hook is not None:
            hook[self._get_name() + str(self.layer) + "_fc2_lif"] = {"tensor": x.detach()}
        x = self.fc2_conv(x.flatten(0, 1))
        if hook is not None:
            hook[self._get_name() + str(self.layer) + "_fc2_lif"]["flops"] = torch.prod(torch.tensor(list(self.fc2_conv.weight.shape))) * x.shape[-1] * x.shape[-2]
        x = self.fc2_bn(x).reshape(T, B, C, H, W).contiguous()

        x = x + identity
        return x, hook


class MS_SSA_Conv(nn.Module):
    def __init__(
        self,
        dim,
        num_heads=8,
        qkv_bias=False,
        qk_scale=None,
        attn_drop=0.0,
        proj_drop=0.0,
        sr_ratio=1,
        mode="direct_xor",
        spike_mode="lif",
        mlif_channels=1,
        scaled=False,
        dvs=False,
        layer=0,
    ):
        super().__init__()
        assert (
            dim % num_heads == 0
        ), f"dim {dim} should be divided by num_heads {num_heads}."
        self.dim = dim
        self.dvs = dvs
        self.num_heads = num_heads
        if dvs:
            self.pool = Erode()
        self.scale = 0.125
        self.q_conv = nn.Conv2d(dim, dim, kernel_size=1, stride=1, bias=False)
        self.q_bn = nn.BatchNorm2d(dim)
        if spike_mode == "lif":
            self.q_lif = MultiStepLIFNode(tau=2.0, detach_reset=True, backend="cupy")
        elif spike_mode == "mlif":
            self.q_lif = mlif.MultiStepMLIFNode(tau=2.0, detach_reset=True, channels=mlif_channels, backend="torch", scaled=scaled)
        elif spike_mode == "fast_mlif":
            self.q_lif = mlif.FastMultiStepMLIFNode(tau=2.0, detach_reset=True, channels=mlif_channels, backend="cupy", scaled=scaled)
        elif spike_mode == "plif":
            self.q_lif = MultiStepParametricLIFNode(
                init_tau=2.0, detach_reset=True, backend="cupy"
            )

        self.k_conv = nn.Conv2d(dim, dim, kernel_size=1, stride=1, bias=False)
        self.k_bn = nn.BatchNorm2d(dim)
        if spike_mode == "lif":
            self.k_lif = MultiStepLIFNode(tau=2.0, detach_reset=True, backend="cupy")
        elif spike_mode == "mlif":
            self.k_lif = mlif.MultiStepMLIFNode(tau=2.0, detach_reset=True, channels=mlif_channels, backend="torch", scaled=scaled)
        elif spike_mode == "fast_mlif":
            self.k_lif = mlif.FastMultiStepMLIFNode(tau=2.0, detach_reset=True, channels=mlif_channels, backend="cupy", scaled=scaled)
        elif spike_mode == "plif":
            self.k_lif = MultiStepParametricLIFNode(
                init_tau=2.0, detach_reset=True, backend="cupy"
            )

        self.v_conv = nn.Conv2d(dim, dim, kernel_size=1, stride=1, bias=False)
        self.v_bn = nn.BatchNorm2d(dim)
        if spike_mode == "lif":
            self.v_lif = MultiStepLIFNode(tau=2.0, detach_reset=True, backend="cupy")
        elif spike_mode == "mlif":
            self.v_lif = mlif.MultiStepMLIFNode(tau=2.0, detach_reset=True, channels=mlif_channels, backend="torch", scaled=scaled)
        elif spike_mode == "fast_mlif":
            self.v_lif = mlif.FastMultiStepMLIFNode(tau=2.0, detach_reset=True, channels=mlif_channels, backend="cupy", scaled=scaled)
        elif spike_mode == "plif":
            self.v_lif = MultiStepParametricLIFNode(
                init_tau=2.0, detach_reset=True, backend="cupy"
            )

        if spike_mode == "lif":
            self.attn_lif = MultiStepLIFNode(
                tau=2.0, v_threshold=0.5, detach_reset=True, backend="cupy"
            )
        elif spike_mode == "mlif":
            self.attn_lif = mlif.MultiStepMLIFNode(
                tau=2.0, v_threshold=0.5, detach_reset=True, channels=mlif_channels, backend="torch", scaled=scaled
            )
        elif spike_mode == "fast_mlif":
            self.attn_lif = mlif.FastMultiStepMLIFNode(
                tau=2.0, v_threshold=0.5, detach_reset=True, channels=mlif_channels, backend="cupy", scaled=scaled
            )
        elif spike_mode == "plif":
            self.attn_lif = MultiStepParametricLIFNode(
                init_tau=2.0, v_threshold=0.5, detach_reset=True, backend="cupy"
            )

        self.talking_heads = nn.Conv1d(
            num_heads, num_heads, kernel_size=1, stride=1, bias=False
        )
        if spike_mode == "lif":
            self.talking_heads_lif = MultiStepLIFNode(
                tau=2.0, v_threshold=0.5, detach_reset=True, backend="cupy"
            )
        elif spike_mode == "mlif":
            self.talking_heads_lif = mlif.MultiStepMLIFNode(
                tau=2.0, v_threshold=0.5, detach_reset=True, channels=mlif_channels, backend="torch", scaled=scaled
            )
        elif spike_mode == "fast_mlif":
            self.talking_heads_lif = mlif.FastMultiStepMLIFNode(
                tau=2.0, v_threshold=0.5, detach_reset=True, channels=mlif_channels, backend="cupy", scaled=scaled
            )
        elif spike_mode == "plif":
            self.talking_heads_lif = MultiStepParametricLIFNode(
                init_tau=2.0, v_threshold=0.5, detach_reset=True, backend="cupy"
            )

        self.proj_conv = nn.Conv2d(dim, dim, kernel_size=1, stride=1)
        self.proj_bn = nn.BatchNorm2d(dim)

        if spike_mode == "lif":
            self.shortcut_lif = MultiStepLIFNode(
                tau=2.0, detach_reset=True, backend="cupy"
            )
        elif spike_mode == "mlif":
            self.shortcut_lif = mlif.MultiStepMLIFNode(
                tau=2.0, detach_reset=True, channels=mlif_channels, backend="torch", scaled=scaled
            )
        elif spike_mode == "fast_mlif":
            self.shortcut_lif = mlif.FastMultiStepMLIFNode(
                tau=2.0, detach_reset=True, channels=mlif_channels, backend="cupy", scaled=scaled
            )
        elif spike_mode == "plif":
            self.shortcut_lif = MultiStepParametricLIFNode(
                init_tau=2.0, detach_reset=True, backend="cupy"
            )

        self.mode = mode
        self.layer = layer
        self.mlif_channels = mlif_channels

    def forward(self, x, hook=None):
        T, B, C, H, W = x.shape
        identity = x
        N = H * W
        x = self.shortcut_lif(x)

        x_for_qkv = x.flatten(0, 1)
        if hook is not None:
            hook[self._get_name() + str(self.layer) + "_kqv_conv"] = {"tensor": x_for_qkv.detach()}
        q_conv_out = self.q_conv(x_for_qkv)
        if hook is not None:
            hook[self._get_name() + str(self.layer) + "_kqv_conv"]["flops"] = 3 * torch.prod(torch.tensor(list(self.q_conv.weight.shape))) * q_conv_out.shape[-1] * q_conv_out.shape[-2] 
        q_conv_out = self.q_bn(q_conv_out).reshape(T, B, C, H, W).contiguous()
        q_conv_out = self.q_lif(q_conv_out)

        q = (
            q_conv_out.flatten(3)
            .transpose(-1, -2)
            .reshape(T, B, N, self.num_heads, C // self.num_heads)
            .permute(0, 1, 3, 2, 4)
            .contiguous()
        )

        k_conv_out = self.k_conv(x_for_qkv)
        k_conv_out = self.k_bn(k_conv_out).reshape(T, B, C, H, W).contiguous()
        k_conv_out = self.k_lif(k_conv_out)
        if self.dvs:
            k_conv_out = self.pool(k_conv_out)

        k = (
            k_conv_out.flatten(3)
            .transpose(-1, -2)
            .reshape(T, B, N, self.num_heads, C // self.num_heads)
            .permute(0, 1, 3, 2, 4)
            .contiguous()
        )

        v_conv_out = self.v_conv(x_for_qkv)
        v_conv_out = self.v_bn(v_conv_out).reshape(T, B, C, H, W).contiguous()
        if self.mode == "sdsa":
            v_conv_out = self.v_lif(v_conv_out)
            if self.dvs:
                v_conv_out = self.pool(v_conv_out)
        
        v = (
            v_conv_out.flatten(3)
            .transpose(-1, -2)
            .reshape(T, B, N, self.num_heads, C // self.num_heads)
            .permute(0, 1, 3, 2, 4)
            .contiguous()
        )  # T B head N C//h

        if self.mode == "sdsa":
            kv = k.mul(v)

            if self.dvs:
                kv = self.pool(kv)
            kv = kv.sum(dim=-2, keepdim=True)
            kv = self.talking_heads_lif(kv)

            x = q.mul(kv)
            if self.dvs:
                x = self.pool(x)

            x = x.transpose(3, 4).reshape(T, B, C, H, W).contiguous()
        
        if hook is not None:
            hook[self._get_name() + str(self.layer) + "_x_after_kqv_conv"] = {"tensor": x.detach()}
        x = (
            self.proj_bn(self.proj_conv(x.flatten(0, 1)))
                .reshape(T, B, C, H, W)
                .contiguous()
        )
        if hook is not None:
            hook[self._get_name() + str(self.layer) + "_x_after_kqv_conv"]["flops"] = torch.prod(torch.tensor(list(self.proj_conv.weight.shape))) * x.shape[-1] * x.shape[-2]
    
        x = x + identity
        return x, v, hook


class MS_Block_Conv(nn.Module):
    def __init__(
        self,
        dim,
        num_heads,
        mlp_ratio=4.0,
        qkv_bias=False,
        qk_scale=None,
        drop=0.0,
        attn_drop=0.0,
        drop_path=0.0,
        norm_layer=nn.LayerNorm,
        sr_ratio=1,
        attn_mode="direct_xor",
        spike_mode="lif",
        mlif_channels=1,
        scaled=False,
        dvs=False,
        layer=0,
    ):
        super().__init__()
        self.attn = MS_SSA_Conv(
            dim,
            num_heads=num_heads,
            qkv_bias=qkv_bias,
            qk_scale=qk_scale,
            attn_drop=attn_drop,
            proj_drop=drop,
            sr_ratio=sr_ratio,
            mode=attn_mode,
            spike_mode=spike_mode,
            mlif_channels=mlif_channels,
            scaled=scaled,
            dvs=dvs,
            layer=layer,
        )
        self.drop_path = DropPath(drop_path) if drop_path > 0.0 else nn.Identity()
        mlp_hidden_dim = int(dim * mlp_ratio)
        self.mlp = MS_MLP_Conv(
            in_features=dim,
            hidden_features=mlp_hidden_dim,
            drop=drop,
            spike_mode=spike_mode,
            mlif_channels=mlif_channels,
            scaled=scaled,
            layer=layer,
        )

    def forward(self, x, hook=None):
        x_attn, attn, hook = self.attn(x, hook=hook)
        x, hook = self.mlp(x_attn, hook=hook)
        return x, attn, hook
