/*
 * Decompiled with CFR 0.152.
 */
package org.openslide;

import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.awt.image.DataBufferInt;
import java.io.Closeable;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import javax.swing.filechooser.FileFilter;
import org.openslide.AssociatedImage;
import org.openslide.OpenSlideDisposedException;
import org.openslide.OpenSlideJNI;

public final class OpenSlide
implements Closeable {
    private static final FileFilter FILE_FILTER = new FileFilter(){

        @Override
        public boolean accept(File f) {
            return f.isDirectory() || OpenSlide.detectVendor(f) != null;
        }

        @Override
        public String getDescription() {
            return "Virtual slide";
        }
    };
    private static final String LIBRARY_VERSION = OpenSlideJNI.openslide_get_version();
    public static final String PROPERTY_NAME_COMMENT = "openslide.comment";
    public static final String PROPERTY_NAME_VENDOR = "openslide.vendor";
    public static final String PROPERTY_NAME_QUICKHASH1 = "openslide.quickhash-1";
    public static final String PROPERTY_NAME_BACKGROUND_COLOR = "openslide.background-color";
    public static final String PROPERTY_NAME_OBJECTIVE_POWER = "openslide.objective-power";
    public static final String PROPERTY_NAME_MPP_X = "openslide.mpp-x";
    public static final String PROPERTY_NAME_MPP_Y = "openslide.mpp-y";
    public static final String PROPERTY_NAME_BOUNDS_X = "openslide.bounds-x";
    public static final String PROPERTY_NAME_BOUNDS_Y = "openslide.bounds-y";
    public static final String PROPERTY_NAME_BOUNDS_WIDTH = "openslide.bounds-width";
    public static final String PROPERTY_NAME_BOUNDS_HEIGHT = "openslide.bounds-height";
    private long osr;
    private final ReadWriteLock lock = new ReentrantReadWriteLock();
    private final long[] levelWidths;
    private final long[] levelHeights;
    private final double[] levelDownsamples;
    private final int levelCount;
    private final Map<String, String> properties;
    private final Map<String, AssociatedImage> associatedImages;
    private final File canonicalFile;
    private final int hashCodeVal;
    final boolean debug = false;
    private int debugThingy = 0;

    public static String detectVendor(File file) {
        return OpenSlideJNI.openslide_detect_vendor(file.getPath());
    }

    public OpenSlide(File file) throws IOException {
        if (!file.exists()) {
            throw new FileNotFoundException(file.toString());
        }
        this.osr = OpenSlideJNI.openslide_open(file.getPath());
        if (this.osr == 0L) {
            throw new IOException(file + ": Not a file that OpenSlide can recognize");
        }
        try {
            this.checkError();
        }
        catch (IOException e) {
            this.dispose();
            throw e;
        }
        this.levelCount = OpenSlideJNI.openslide_get_level_count(this.osr);
        this.levelWidths = new long[this.levelCount];
        this.levelHeights = new long[this.levelCount];
        this.levelDownsamples = new double[this.levelCount];
        for (int i = 0; i < this.levelCount; ++i) {
            long[] lArray = new long[2];
            OpenSlideJNI.openslide_get_level_dimensions(this.osr, i, lArray);
            this.levelWidths[i] = lArray[0];
            this.levelHeights[i] = lArray[1];
            this.levelDownsamples[i] = OpenSlideJNI.openslide_get_level_downsample(this.osr, i);
        }
        HashMap<String, String> props = new HashMap<String, String>();
        for (String s : OpenSlideJNI.openslide_get_property_names(this.osr)) {
            props.put(s, OpenSlideJNI.openslide_get_property_value(this.osr, s));
        }
        this.properties = Collections.unmodifiableMap(props);
        HashMap<String, AssociatedImage> hashMap = new HashMap<String, AssociatedImage>();
        for (String s : OpenSlideJNI.openslide_get_associated_image_names(this.osr)) {
            hashMap.put(s, new AssociatedImage(s, this));
        }
        this.associatedImages = Collections.unmodifiableMap(hashMap);
        this.canonicalFile = file.getCanonicalFile();
        String quickhash1 = this.getProperties().get(PROPERTY_NAME_QUICKHASH1);
        this.hashCodeVal = quickhash1 != null ? (int)Long.parseLong(quickhash1.substring(0, 8), 16) : this.canonicalFile.hashCode();
        try {
            this.checkError();
        }
        catch (IOException e) {
            this.dispose();
            throw e;
        }
    }

    private void checkError() throws IOException {
        String msg = OpenSlideJNI.openslide_get_error(this.osr);
        if (msg != null) {
            throw new IOException(msg);
        }
    }

    public void dispose() {
        Lock wl = this.lock.writeLock();
        wl.lock();
        try {
            if (this.osr != 0L) {
                OpenSlideJNI.openslide_close(this.osr);
                this.osr = 0L;
            }
        }
        finally {
            wl.unlock();
        }
    }

    public int getLevelCount() {
        return this.levelCount;
    }

    private void checkDisposed() {
        if (this.osr == 0L) {
            throw new OpenSlideDisposedException();
        }
    }

    public long getLevel0Width() {
        return this.levelWidths[0];
    }

    public long getLevel0Height() {
        return this.levelHeights[0];
    }

    public long getLevelWidth(int level) {
        return this.levelWidths[level];
    }

    public long getLevelHeight(int level) {
        return this.levelHeights[level];
    }

    public void paintRegionOfLevel(Graphics2D g, int dx, int dy, int sx, int sy, int w, int h, int level) throws IOException {
        this.paintRegion(g, dx, dy, sx, sy, w, h, this.levelDownsamples[level]);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void paintRegionARGB(int[] dest, long x, long y, int level, int w, int h) throws IOException {
        if ((long)w * (long)h > (long)dest.length) {
            throw new ArrayIndexOutOfBoundsException("Size of data (" + dest.length + ") is less than w * h");
        }
        if (w < 0 || h < 0) {
            throw new IllegalArgumentException("w and h must be nonnegative");
        }
        Lock rl = this.lock.readLock();
        rl.lock();
        try {
            this.checkDisposed();
            OpenSlideJNI.openslide_read_region(this.osr, dest, x, y, level, w, h);
            this.checkError();
        }
        finally {
            rl.unlock();
        }
    }

    public void paintRegion(Graphics2D g, int dx, int dy, long sx, long sy, int w, int h, double downsample) throws IOException {
        if (downsample < 1.0) {
            throw new IllegalArgumentException("downsample (" + downsample + ") must be >= 1.0");
        }
        int level = this.getBestLevelForDownsample(downsample);
        double levelDS = this.levelDownsamples[level];
        double relativeDS = downsample / levelDS;
        long baseX = (long)(downsample * (double)sx);
        long baseY = (long)(downsample * (double)sy);
        long levelX = (long)(relativeDS * (double)sx);
        long levelY = (long)(relativeDS * (double)sy);
        int levelW = (int)Math.round(relativeDS * (double)w);
        int levelH = (int)Math.round(relativeDS * (double)h);
        levelW = (int)Math.min((long)levelW, this.getLevelWidth(level) - levelX);
        levelH = (int)Math.min((long)levelH, this.getLevelHeight(level) - levelY);
        w = (int)Math.round((double)levelW / relativeDS);
        h = (int)Math.round((double)levelH / relativeDS);
        if (levelW <= 0 || levelH <= 0) {
            return;
        }
        BufferedImage img = new BufferedImage(levelW, levelH, 3);
        int[] data = ((DataBufferInt)img.getRaster().getDataBuffer()).getData();
        this.paintRegionARGB(data, baseX, baseY, level, img.getWidth(), img.getHeight());
        g.drawImage(img, dx, dy, w, h, null);
    }

    public BufferedImage createThumbnailImage(int x, int y, long w, long h, int maxSize, int bufferedImageType) throws IOException {
        double ds = w > h ? (double)w / (double)maxSize : (double)h / (double)maxSize;
        if (ds < 1.0) {
            ds = 1.0;
        }
        int sw = (int)((double)w / ds);
        int sh = (int)((double)h / ds);
        int sx = (int)((double)x / ds);
        int sy = (int)((double)y / ds);
        BufferedImage result = new BufferedImage(sw, sh, bufferedImageType);
        Graphics2D g = result.createGraphics();
        this.paintRegion(g, 0, 0, sx, sy, sw, sh, ds);
        g.dispose();
        return result;
    }

    public BufferedImage createThumbnailImage(int x, int y, long w, long h, int maxSize) throws IOException {
        return this.createThumbnailImage(x, y, w, h, maxSize, 1);
    }

    public BufferedImage createThumbnailImage(int maxSize) throws IOException {
        return this.createThumbnailImage(0, 0, this.getLevel0Width(), this.getLevel0Height(), maxSize);
    }

    public double getLevelDownsample(int level) {
        return this.levelDownsamples[level];
    }

    public int getBestLevelForDownsample(double downsample) {
        if (downsample < this.levelDownsamples[0]) {
            return 0;
        }
        for (int i = 1; i < this.levelCount; ++i) {
            if (!(downsample < this.levelDownsamples[i])) continue;
            return i - 1;
        }
        return this.levelCount - 1;
    }

    public Map<String, String> getProperties() {
        return this.properties;
    }

    public Map<String, AssociatedImage> getAssociatedImages() {
        return this.associatedImages;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    BufferedImage getAssociatedImage(String name) throws IOException {
        Lock rl = this.lock.readLock();
        rl.lock();
        try {
            this.checkDisposed();
            long[] dim = new long[2];
            OpenSlideJNI.openslide_get_associated_image_dimensions(this.osr, name, dim);
            this.checkError();
            if (dim[0] == -1L) {
                throw new IOException("Failure reading associated image");
            }
            BufferedImage img = new BufferedImage((int)dim[0], (int)dim[1], 3);
            int[] data = ((DataBufferInt)img.getRaster().getDataBuffer()).getData();
            OpenSlideJNI.openslide_read_associated_image(this.osr, name, data);
            this.checkError();
            BufferedImage bufferedImage = img;
            return bufferedImage;
        }
        finally {
            rl.unlock();
        }
    }

    public static String getLibraryVersion() {
        return LIBRARY_VERSION;
    }

    public static FileFilter getFileFilter() {
        return FILE_FILTER;
    }

    public int hashCode() {
        return this.hashCodeVal;
    }

    public boolean equals(Object obj) {
        if (obj == this) {
            return true;
        }
        if (obj instanceof OpenSlide) {
            OpenSlide os2 = (OpenSlide)obj;
            String quickhash1 = this.getProperties().get(PROPERTY_NAME_QUICKHASH1);
            String os2_quickhash1 = os2.getProperties().get(PROPERTY_NAME_QUICKHASH1);
            if (quickhash1 != null && os2_quickhash1 != null) {
                return quickhash1.equals(os2_quickhash1);
            }
            if (quickhash1 == null && os2_quickhash1 == null) {
                return this.canonicalFile.equals(os2.canonicalFile);
            }
            return false;
        }
        return false;
    }

    @Override
    public void close() {
        this.dispose();
    }
}

