from pathlib import Path
from collections import defaultdict
from typing import Dict, List, Set, Tuple
import os
def analyze_directory_structure(root_path: str) -> Tuple[Dict[str, int], int, int]:
    """
    Analyze directory structure to calculate various metrics about models and files.
    
    Args:
        root_path (str): Path to the root directory to analyze
        
    Returns:
        Tuple containing:
        - Dictionary of model names and their file counts
        - Number of folders without llama-70b
        - Number of folders where llama-70b has less than 6 files
    """
    root = Path(root_path)
    model_file_counts = defaultdict(int)
    folders_without_llama = 0
    folders_llama_less_than_6 = 0
    
    # Get all immediate subdirectories in root
    main_folders = [d for d in root.iterdir() if d.is_dir()]
    
    for main_folder in main_folders:
        if main_folder.name != 'simulation':
            continue
        # Get all sub-folders (which contain model folders)
        sub_folders = [d for d in main_folder.iterdir() if d.is_dir()]
        # Skip non-simulation folders
        
        for sub_folder in sub_folders:
            # Track if we found llama-70b in this folder's models
            found_llama = False
            llama_file_count = 0
            # Get model folders
            model_folders = [d for d in sub_folder.iterdir() if d.is_dir()]
            
            for model_folder in model_folders:
                # Count files in model folder
                file_count = len([f for f in model_folder.glob('*') if f.is_file()])
                model_name = model_folder.name
                model_file_counts[model_name] += file_count
                
                # Check if this is a llama-70b folder
                if 'llama-3.1-70b' in model_name.lower():
                    found_llama = True
                    llama_file_count = file_count
                    if llama_file_count < 6:
                        folders_llama_less_than_6 += 1
                        # print(f"Folder {sub_folder.name} has {llama_file_count} llama-3.1-70b files")


    return model_file_counts, folders_without_llama, folders_llama_less_than_6

def main():
    # Example usage
    root_path = os.path.join("result")
    
    try:
        model_counts, no_llama_count, llama_less_than_6 = analyze_directory_structure(root_path)
        
        print("\nFile counts per model:")
        for model, count in sorted(model_counts.items()):
            print(f"{model}: {count} files")
            
        print(f"\nNumber of folders without llama-70b: {no_llama_count}")
        print(f"Number of folders where llama-70b has less than 6 files: {llama_less_than_6}")
        
    except Exception as e:
        print(f"An error occurred: {str(e)}")

if __name__ == "__main__":
    main()
