# for one use only

#!/usr/bin/env python3
import os
import re
import pandas as pd

def main():
    """
    Change all occurrences of chat_round_order=5 to chat_round_order=4
    in opinion_memory_{eval_model_name}_{version}.csv files
    for data prefixes starting with 2025(03|04).
    """
    # Base result directory
    base_dir = "../../result/eval/human_llm"
    # File pattern to match
    file_pattern = r"^opinion_memory_.*_v\d+\.csv$"
    # Data prefix pattern
    data_prefix_pattern = r"^2025(03|04)"
    
    print("Starting to process opinion memory files...")
    print(f"Looking for data prefixes in {base_dir}")
    
    # Get all data prefix directories
    data_prefixes = []
    for entry in os.listdir(base_dir):
        entry_path = os.path.join(base_dir, entry)
        if os.path.isdir(entry_path) and re.match(data_prefix_pattern, entry):
            data_prefixes.append(entry)
    
    print(f"Found {len(data_prefixes)} matching data prefixes: {data_prefixes}")
    
    total_files_processed = 0
    total_changes_made = 0
    
    # Process each data prefix directory
    for data_prefix in data_prefixes:
        data_prefix_path = os.path.join(base_dir, data_prefix)
        print(f"Examining data prefix directory: {data_prefix_path}")
        
        # Process each model directory
        for model_name in os.listdir(data_prefix_path):
            model_dir = os.path.join(data_prefix_path, model_name)
            if os.path.isdir(model_dir):
                print(f"  Checking model directory: {model_name}")
                # Check all files in the model directory
                for filename in os.listdir(model_dir):
                    print(f"    Checking file: {filename}")
                    if re.match(file_pattern, filename):
                        file_path = os.path.join(model_dir, filename)
                        print(f"    Found matching file: {filename}")
                        changes = process_file(file_path)
                        total_files_processed += 1
                        total_changes_made += changes
                        print(f"    Processed {file_path}: {changes} changes made")
    
    print(f"Completed processing {total_files_processed} files with {total_changes_made} total changes")

def process_file(file_path):
    """
    Process a CSV file to replace chat_round_order=5 with chat_round_order=4
    
    Args:
        file_path: Path to the CSV file
    
    Returns:
        Number of changes made
    """
    # Check if file exists
    if not os.path.exists(file_path):
        print(f"Warning: File {file_path} does not exist")
        return 0
    
    try:
        # Read the file
        df = pd.read_csv(file_path)
        
        # Print column names to understand the structure
        print(f"Columns in {file_path}: {df.columns.tolist()}")
        
        # Check for various possible column names related to chat rounds
        round_columns = [col for col in df.columns if 'round' in col.lower()]
        if round_columns:
            print(f"Found round-related columns: {round_columns}")
            
            # Check "chat_round_order" column if it exists
            if 'chat_round_order' in df.columns:
                # Count occurrences of chat_round_order=5
                num_changes = (df['chat_round_order'] == 5).sum()
                
                if num_changes > 0:
                    print(f"Found {num_changes} rows with chat_round_order=5")
                    # Replace 5 with 4
                    df.loc[df['chat_round_order'] == 5, 'chat_round_order'] = 4
                    
                    # Save the file
                    df.to_csv(file_path, index=False)
                    
                    return num_changes
                else:
                    print("No rows with chat_round_order=5 found")
            else:
                print("'chat_round_order' not found, but other round columns exist")
        else:
            print(f"No round-related columns found in {file_path}")
        
        return 0
    
    except Exception as e:
        print(f"Error processing {file_path}: {e}")
        return 0

if __name__ == "__main__":
    main()
