import os
import pandas as pd
from docx import Document
from docx.shared import Inches, Pt
from docx.oxml.ns import qn
from docx.oxml import OxmlElement
import cairosvg
# to use cairosvg: export DYLD_FALLBACK_LIBRARY_PATH=/opt/homebrew/opt/cairo/lib 

PROJECT_ROOT = os.path.abspath(os.path.join(os.path.dirname(__file__), "..", ".."))
EVAL_PATH = os.path.join(PROJECT_ROOT, "result", "eval", "group_level")
OUTPUT_PATH = os.path.join(EVAL_PATH, "group_level_summary.docx")

TOPICS = [
    "A_\"body_cleanse,\"_in_which_you_consume_only_particular_kinds_of_nutrients_over_1-3_days,_helps_your_body_to_eliminate_toxins",
    "Angels_are_real",
    "Everything_that_happens_can_eventually_be_explained_by_science",
    "Regular_fasting_will_improve_your_health",
    "The_position_of_the_planets_at_the_time_of_your_birth_can_influence_your_personality",
    "The_United_States_has_the_highest_federal_income_tax_rate_of_any_Western_country",
    "The_US_deficit_increased_after_President_Obama_was_elected",
]

MODEL_NAME = "gpt-4o-mini-2024-07-18"

# Define report sections
SECTIONS = [
    (
        "Human Data: The Opinion Diversity (Change in the Standard Deviation of the Group’s Opinions)",
        "human",
        "summary_stance_std_errorbar.svg"
    ),
    (
        "Human Data: The Opinion Bias (Change in the Average of Group’s Opinion)",
        "human",
        "summary_stance_bias_errorbar.svg"
    ),
    (
        "LLM Simulation: The Opinion Diversity (Change in the Standard Deviation of the Group’s Opinions)",
        "simulation",
        "summary_stance_std_errorbar.svg"
    ),
    (
        "LLM Simulation: The Opinion Bias (Change in the Average of Group’s Opinion)",
        "simulation",
        "summary_stance_bias_errorbar.svg"
    ),
]

def extract_experiment_labels():
    """Extract unique 'label' values from stance_stats.csv under each topic."""
    label_set = set()

    for topic in TOPICS:
        csv_path = os.path.join(EVAL_PATH, topic, "human", "tweet", "stance_stats.csv")
        if os.path.exists(csv_path):
            try:
                df = pd.read_csv(csv_path)
                if 'label' in df.columns:
                    label_set.update(df['label'].dropna().unique())
            except Exception as e:
                print(f"[Error reading {csv_path}]: {str(e)}")

    return sorted(label_set)

def append_metadata(doc, simulation_version="v0"):
    """Append experiment metadata at the end of the doc."""
    doc.add_page_break()
    doc.add_heading("Appendix: Experiment Metadata", level=1)

    # Simulation version
    doc.add_heading("Simulation Version", level=2)
    doc.add_paragraph(simulation_version)

    # Experiment labels
    doc.add_heading("Experiment Labels", level=2)
    labels = extract_experiment_labels()
    for label in labels:
        doc.add_paragraph(label, style='List Bullet')

def convert_svg_if_needed(svg_path, png_path):
    """Convert SVG to PNG if PNG doesn't exist or is older than the SVG."""
    if (
        not os.path.exists(png_path) or
        os.path.getmtime(svg_path) > os.path.getmtime(png_path)
    ):
        cairosvg.svg2png(url=svg_path, write_to=png_path)

def insert_images_in_rows(doc, image_paths, images_per_row=3):
    """Insert images in a table with N images per row using safe size."""
    table = doc.add_table(rows=1, cols=images_per_row)
    row_cells = table.rows[0].cells
    for i, image_path in enumerate(image_paths):
        if i % images_per_row == 0 and i != 0:
            row_cells = table.add_row().cells
        cell = row_cells[i % images_per_row]
        paragraph = cell.paragraphs[0]
        run = paragraph.add_run()
        run.add_picture(image_path, width=Inches(1.94), height=Inches(1.61))

def generate_summary_docx():
    doc = Document()
    doc.add_heading("Group Level Summary Report", 0)

    for section_title, mode, filename in SECTIONS:
        doc.add_heading(section_title, level=1)
        image_paths = []

        for topic in TOPICS:
            svg_path = os.path.join(EVAL_PATH, topic, mode, "tweet", MODEL_NAME, filename)
            png_path = svg_path.replace(".svg", ".png")

            if os.path.exists(svg_path):
                try:
                    convert_svg_if_needed(svg_path, png_path)
                    image_paths.append(png_path)
                except Exception as e:
                    print(f"[Error converting image for {topic}]: {str(e)}")
            else:
                print(f"[Missing] {svg_path}")

        if image_paths:
            insert_images_in_rows(doc, image_paths)

    os.makedirs(EVAL_PATH, exist_ok=True)
    append_metadata(doc, simulation_version="v0")
    doc.save(OUTPUT_PATH)
    print(f"✅ Report saved to: {OUTPUT_PATH}")

if __name__ == "__main__":
    generate_summary_docx()