#!/usr/bin/env python3
"""
Script to generate a Markdown file showing all opinion topic plots
sequentially with headings for simulation versions and topics.
"""

import os
import re
from collections import defaultdict

def main(path: str):
    # Directory containing the plot files
    plot_dir = path
    
    # Dictionary to store files grouped by topic
    topic_files = defaultdict(list)
    
    # Pattern to match plot files and extract topic and version
    pattern = re.compile(r'opinion_plot_(.+)_gpt-4o-mini-2024-07-18_v(\d+)\.svg')
    
    # Scan directory for plot files
    for filename in os.listdir(plot_dir):
        if filename.endswith('.svg') and filename.startswith('opinion_plot_'):
            match = pattern.match(filename)
            if match:
                topic, version = match.groups()
                topic_files[topic].append((int(version), filename))
    
    # Sort topics alphabetically, but put "All" at the beginning if it exists
    topics = sorted(topic_files.keys())
    if "All" in topics:
        topics.remove("All")
        topics.insert(0, "All")
    
    # Sort files within each topic by version
    for topic in topic_files:
        topic_files[topic].sort()  # Sort by version
    
    # Generate Markdown content
    markdown_content = "# Opinion Topic Plots for GPT-4o-mini-2024-07-18\n\n"
    
    for topic in topics:
        markdown_content += f"## Topic: {topic}\n\n"
        
        for version, filename in topic_files[topic]:
            markdown_content += f"### Simulation v{version}\n\n"
            # URL encode spaces in the filename for proper URL formatting
            url_encoded_filename = filename.replace(" ", "%20")
            markdown_content += f"![{topic} v{version}](./{url_encoded_filename})\n\n"
    
    # Write to Markdown file
    output_path = os.path.join(plot_dir, "opinion_plots.md")
    with open(output_path, 'w') as f:
        f.write(markdown_content)
    
    print(f"Markdown file generated at: {output_path}")

if __name__ == "__main__":
    main("../../result/eval/opinion_topics/gpt-4o-mini-2024-07-18")
