import os
import pandas as pd
from docx import Document
from docx.shared import Inches
import ctypes
import ctypes.util

def resolve_cairo_path():
    # 1) Try the system’s library resolver (portable)
    lib = ctypes.util.find_library("cairo")
    if lib:
        return lib  # e.g., "libcairo.so.2" or absolute path

    # 2) Try common absolute paths (Linux)
    candidates = [
        "/lib64/libcairo.so.2",
        "/usr/lib64/libcairo.so.2",
        "/usr/lib/x86_64-linux-gnu/libcairo.so.2",
        "/lib64/libcairo.so",
        "/lib/x86_64-linux-gnu/libcairo.so.2",
    ]
    for p in candidates:
        if os.path.exists(p):
            return p

    # 3) Optional: allow override via env var
    env_path = os.environ.get("LIBCAIRO_PATH")
    if env_path and os.path.exists(env_path):
        return env_path

    return None

lib_path = resolve_cairo_path()
if not lib_path:
    raise FileNotFoundError(
        "Cairo library not found. Install libcairo (e.g., apt/yum), or set LIBCAIRO_PATH to the .so path."
    )

# Load it so downstream libs (e.g., cairosvg) can find symbols
ctypes.CDLL(lib_path)
print(f"[cairo] Using {lib_path}")

import cairosvg

# Set DYLD_FALLBACK_LIBRARY_PATH before running if needed:
# export DYLD_FALLBACK_LIBRARY_PATH=/opt/homebrew/opt/cairo/lib

PROJECT_ROOT = os.path.abspath(os.path.join(os.path.dirname(__file__), "..", ".."))
EVAL_PATH = os.path.join(PROJECT_ROOT, "result", "group_level_eval", "base_analysis")
OUTPUT_PATH = os.path.join(EVAL_PATH, "base_model_summary.docx")
OUTPUT_PATH_PUBLICATION = os.path.join(EVAL_PATH, "base_model_summary_publication.docx")
OUTPUT_PATH_SPLIT = os.path.join(EVAL_PATH, "base_model_summary_split.docx")


TOPICS = [
    "A_\"body_cleanse,\"_in_which_you_consume_only_particular_kinds_of_nutrients_over_1-3_days,_helps_your_body_to_eliminate_toxins",
    "Angels_are_real",
    "Everything_that_happens_can_eventually_be_explained_by_science",
    "Everything_that_happens_can_eventually_be_explained_by_science_reversed",
    "Regular_fasting_will_improve_your_health",
    "The_position_of_the_planets_at_the_time_of_your_birth_can_influence_your_personality",
    "The_United_States_has_the_highest_federal_income_tax_rate_of_any_Western_country",
    "The_US_deficit_increased_after_President_Obama_was_elected",
]

SECTIONS_PUBLICATION = [
    (
        "Human Data: The Opinion Diversity (Change in the Standard Deviation of the Group’s Opinions)",
        "summary_std_errorbar_publication.svg",
        "human"
    ),
    (
        "Human Data: The Opinion Bias (Change in the Average of Group’s Opinion)",
        "summary_bias_errorbar_publication.svg",
        "human"
    ),
    (
        "LLM Simulation: The Opinion Diversity (Change in the Standard Deviation of the Group’s Opinions)",
        "summary_std_errorbar_publication.svg",
        "gpt-4o-mini-2024-07-18"
    ),
    (
        "LLM Simulation: The Opinion Bias (Change in the Average of Group’s Opinion)",
        "summary_bias_errorbar_publication.svg",
        "gpt-4o-mini-2024-07-18"
    ),
    (
        "Human vs. LLM: The Opinion Diversity (Change in the Standard Deviation of the Group’s Opinions)",
        "combined_summary_std_errorbar_publication.svg",
        None
    ),
    (
        "Human vs. LLM: The Opinion Bias (Change in the Average of Group’s Opinion)",
        "combined_summary_bias_errorbar_publication.svg",
        None
    ),
    (
        "Human vs. LLM: Slider-Based Opinion Diversity (Initial ➜ Post)",
        "combined_summary_std_errorbar_slider_publication.svg",
        None
    ),
    (
        "Human vs. LLM: Slider-Based Opinion Bias (Initial ➜ Post)",
        "combined_summary_bias_errorbar_slider_publication.svg",
        None
    ),
    (
        "Human vs. LLM: Stance-Based Opinion Diversity (Tweet 1 ➜ Tweet 3)",
        "combined_summary_std_errorbar_stance_publication.svg",
        None
    ),
    (
        "Human vs. LLM: Stance-Based Opinion Bias (Tweet 1 ➜ Tweet 3)",
        "combined_summary_bias_errorbar_stance_publication.svg",
        None
    ),
]

SECTIONS = [
    (
        "Human Data: The Opinion Diversity (Change in the Standard Deviation of the Group’s Opinions)",
        "summary_std_errorbar.svg",
        "human"
    ),
    (
        "Human Data: The Opinion Bias (Change in the Average of Group’s Opinion)",
        "summary_bias_errorbar.svg",
        "human"
    ),
    (
        "LLM Simulation: The Opinion Diversity (Change in the Standard Deviation of the Group’s Opinions)",
        "summary_std_errorbar.svg",
        "gpt-4o-mini-2024-07-18"
    ),
    (
        "LLM Simulation: The Opinion Bias (Change in the Average of Group’s Opinion)",
        "summary_bias_errorbar.svg",
        "gpt-4o-mini-2024-07-18"
    ),
    (
        "Human vs. LLM: The Opinion Diversity (Change in the Standard Deviation of the Group’s Opinions)",
        "combined_summary_std_errorbar.svg",
        None  # combined section
    ),
    (
        "Human vs. LLM: The Opinion Bias (Change in the Average of Group’s Opinion)",
        "combined_summary_bias_errorbar.svg",
        None  # combined section
    ),
    (
        "Human vs. LLM: Slider-Based Opinion Diversity (Initial ➜ Post)",
        "combined_summary_std_errorbar_slider.svg",
        None
    ),
    (
        "Human vs. LLM: Slider-Based Opinion Bias (Initial ➜ Post)",
        "combined_summary_bias_errorbar_slider.svg",
        None
    ),
    (
        "Human vs. LLM: Stance-Based Opinion Diversity (Tweet 1 ➜ Tweet 3)",
        "combined_summary_std_errorbar_stance.svg",
        None
    ),
    (
        "Human vs. LLM: Stance-Based Opinion Bias (Tweet 1 ➜ Tweet 3)",
        "combined_summary_bias_errorbar_stance.svg",
        None
    ),
]

SECTIONS_SPLIT = [
    (
        "Stance-Based",
        [
            (
                "Human",
                "human",
                [
                    ("The Opinion Diversity", "summary_std_errorbar_stance"),
                    ("The Opinion Bias", "summary_bias_errorbar_stance"),
                ],
                ["opinion", "tweet"],
            ),
            (
                "Model",
                "gpt-4o-mini-2024-07-18",
                [
                    ("The Opinion Diversity", "summary_std_errorbar_stance"),
                    ("The Opinion Bias", "summary_bias_errorbar_stance"),
                ],
                ["opinion", "tweet"],
            ),
            (
                "Combined",
                None,
                [
                    ("The Opinion Diversity", "combined_summary_std_errorbar_stance"),
                    ("The Opinion Bias", "combined_summary_bias_errorbar_stance"),
                ],
                ["opinion", "tweet"],
            ),
        ],
    ),
    (
        "Slider-Based",
        [
            (
                "Human",
                "human",
                [
                    ("The Opinion Diversity", "summary_std_errorbar_slider"),
                    ("The Opinion Bias", "summary_bias_errorbar_slider"),
                ],
                ["opinion"],
            ),
            (
                "Model",
                "gpt-4o-mini-2024-07-18",
                [
                    ("The Opinion Diversity", "summary_std_errorbar_slider"),
                    ("The Opinion Bias", "summary_bias_errorbar_slider"),
                ],
                ["opinion"],
            ),
            (
                "Combined",
                None,
                [
                    ("The Opinion Diversity", "combined_summary_std_errorbar_slider"),
                    ("The Opinion Bias", "combined_summary_bias_errorbar_slider"),
                ],
                ["opinion"],
            ),
        ],
    ),
]

def generate_transition_matrix_section(doc):
    doc.add_heading("Opinion Transition Matrices (Initial ➜ Post, Tweet 1 ➜ Tweet 3)", level=1)

    for topic in TOPICS:
        doc.add_heading(topic.replace("_", " "), level=2)

        # --- 1. Opinion: human vs model ---
        opinion_row = doc.add_table(rows=1, cols=2).rows[0].cells
        for i, source in enumerate(["human", "gpt-4o-mini-2024-07-18"]):
            svg_path = os.path.join(EVAL_PATH, topic, source, "transition_matrix.svg")
            png_path = svg_path.replace(".svg", ".png")
            if os.path.exists(svg_path):
                try:
                    convert_svg_if_needed(svg_path, png_path)
                    paragraph = opinion_row[i].paragraphs[0]
                    run = paragraph.add_run()
                    run.add_picture(png_path, width=Inches(2.7))
                    opinion_row[i].add_paragraph(f"{source} (opinion)")
                except Exception as e:
                    opinion_row[i].text = f"[Error: {e}]"
            else:
                opinion_row[i].text = f"[Missing: {svg_path}]"

        # --- 2. Tweet: human vs model ---
        tweet_row = doc.add_table(rows=1, cols=2).rows[0].cells
        for i, source in enumerate(["human", "gpt-4o-mini-2024-07-18"]):
            svg_path = os.path.join(EVAL_PATH, topic, source, "tweet_transition_matrix.svg")
            png_path = svg_path.replace(".svg", ".png")
            if os.path.exists(svg_path):
                try:
                    convert_svg_if_needed(svg_path, png_path)
                    paragraph = tweet_row[i].paragraphs[0]
                    run = paragraph.add_run()
                    run.add_picture(png_path, width=Inches(2.7))
                    tweet_row[i].add_paragraph(f"{source} (tweet)")
                except Exception as e:
                    tweet_row[i].text = f"[Error: {e}]"
            else:
                tweet_row[i].text = f"[Missing: {svg_path}]"

        # --- 3. Opinion Diff ---
        opinion_diff_svg = os.path.join(EVAL_PATH, topic, "transition_matrix_diff.svg")
        opinion_diff_png = opinion_diff_svg.replace(".svg", ".png")
        if os.path.exists(opinion_diff_svg):
            try:
                convert_svg_if_needed(opinion_diff_svg, opinion_diff_png)
                doc.add_paragraph("Model − Human Difference (opinion):")
                run = doc.add_paragraph().add_run()
                run.add_picture(opinion_diff_png, width=Inches(5.5))
            except Exception as e:
                doc.add_paragraph(f"[Error inserting opinion diff: {e}]")
        else:
            doc.add_paragraph(f"[Missing opinion diff: {opinion_diff_svg}]")

        # --- 4. Tweet Diff ---
        tweet_diff_svg = os.path.join(EVAL_PATH, topic, "tweet_transition_matrix_diff.svg")
        tweet_diff_png = tweet_diff_svg.replace(".svg", ".png")
        if os.path.exists(tweet_diff_svg):
            try:
                convert_svg_if_needed(tweet_diff_svg, tweet_diff_png)
                doc.add_paragraph("Model − Human Difference (tweet):")
                run = doc.add_paragraph().add_run()
                run.add_picture(tweet_diff_png, width=Inches(5.5))
            except Exception as e:
                doc.add_paragraph(f"[Error inserting tweet diff: {e}]")
        else:
            doc.add_paragraph(f"[Missing tweet diff: {tweet_diff_svg}]")

def convert_svg_if_needed(svg_path, png_path):
    if (
        not os.path.exists(png_path) or
        os.path.getmtime(svg_path) > os.path.getmtime(png_path)
    ):
        cairosvg.svg2png(url=svg_path, write_to=png_path)

def insert_images_in_rows(doc, image_paths, images_per_row=3):
    table = doc.add_table(rows=1, cols=images_per_row)
    row_cells = table.rows[0].cells
    for i, image_path in enumerate(image_paths):
        if i % images_per_row == 0 and i != 0:
            row_cells = table.add_row().cells
        cell = row_cells[i % images_per_row]
        paragraph = cell.paragraphs[0]
        run = paragraph.add_run()
        run.add_picture(image_path, width=Inches(1.94), height=Inches(1.61))

def append_metadata(doc):
    doc.add_page_break()
    doc.add_heading("Appendix: Metadata", level=1)

    doc.add_heading("Topics and Datasets", level=2)
    for topic in TOPICS:
        doc.add_paragraph(topic, style='List Bullet')
        csv_path = os.path.join(EVAL_PATH, topic, "filtered_depth.csv")
        try:
            if os.path.exists(csv_path):
                df = pd.read_csv(csv_path)
                time_stamps = sorted(df["time_stamp"].dropna().unique())
                for ts in time_stamps:
                    doc.add_paragraph(f"↳ {ts}", style='List Bullet 2')
            else:
                doc.add_paragraph("↳ [Missing filtered_depth.csv]", style='List Bullet 2')
        except Exception as e:
            doc.add_paragraph(f"↳ [Error: {e}]", style='List Bullet 2')
    
    doc.add_heading("Simulation Version", level=2)
    doc.add_paragraph("v0")

def generate_aggregated_transition_section(doc):
    doc.add_heading("Aggregated Opinion Transition Matrices (Initial ➜ Post, Tweet 1 ➜ Tweet 3)", level=1)

    # --- Opinion Matrices ---
    opinion_entries = [
        ("Aggregated Human (opinion)", "aggregated_transition_human.svg"),
        ("Aggregated gpt-4o-mini-2024-07-18 (opinion)", "aggregated_transition_gpt-4o-mini-2024-07-18.svg"),
        ("Model − Human Difference (opinion)", "aggregated_transition_diff.svg")
    ]

    for label, svg_file in opinion_entries:
        svg_path = os.path.join(EVAL_PATH, svg_file)
        png_path = svg_path.replace(".svg", ".png")

        if os.path.exists(svg_path):
            try:
                convert_svg_if_needed(svg_path, png_path)
                doc.add_paragraph(label + ":")
                run = doc.add_paragraph().add_run()
                run.add_picture(png_path, width=Inches(5.5))
            except Exception as e:
                doc.add_paragraph(f"[Error inserting {label}: {e}]")
        else:
            doc.add_paragraph(f"[Missing aggregated image: {svg_path}]")

    # --- Tweet Matrices ---
    tweet_entries = [
        ("Aggregated Human (tweet)", "aggregated_tweet_transition_human.svg"),
        ("Aggregated gpt-4o-mini-2024-07-18 (tweet)", "aggregated_tweet_transition_gpt-4o-mini-2024-07-18.svg"),
        ("Model − Human Difference (tweet)", "aggregated_tweet_transition_diff.svg")
    ]

    for label, svg_file in tweet_entries:
        svg_path = os.path.join(EVAL_PATH, svg_file)
        png_path = svg_path.replace(".svg", ".png")

        if os.path.exists(svg_path):
            try:
                convert_svg_if_needed(svg_path, png_path)
                doc.add_paragraph(label + ":")
                run = doc.add_paragraph().add_run()
                run.add_picture(png_path, width=Inches(5.5))
            except Exception as e:
                doc.add_paragraph(f"[Error inserting {label}: {e}]")
        else:
            doc.add_paragraph(f"[Missing aggregated tweet image: {svg_path}]")

def generate_split_errorbar_section(doc):
    def insert_grid_images(image_paths):
        if image_paths:
            insert_images_in_rows(doc, image_paths, images_per_row=3)
        else:
            doc.add_paragraph("[Missing images]")

    for section_label, groups in SECTIONS_SPLIT:
        doc.add_page_break()
        doc.add_heading(section_label, level=1)

        for group_label, source, metrics, subtags in groups:
            doc.add_heading(f"{group_label}:", level=2)

            for metric_label, filename_prefix in metrics:
                doc.add_heading(metric_label, level=3)

                for subtag in subtags:
                    doc.add_paragraph(f"{subtag.capitalize()} Trajectory", style="Heading 4")

                    image_paths = []
                    for topic in TOPICS:
                        if source is None:
                            svg_path = os.path.join(EVAL_PATH, topic, f"{filename_prefix}_{subtag}.svg")
                        else:
                            svg_path = os.path.join(EVAL_PATH, topic, source, f"{filename_prefix}_{subtag}.svg")

                        png_path = svg_path.replace(".svg", ".png")
                        if os.path.exists(svg_path):
                            convert_svg_if_needed(svg_path, png_path)
                            image_paths.append(png_path)

                    insert_grid_images(image_paths)

    # Aggregated section
    doc.add_page_break()
    doc.add_heading("Aggregated Plots", level=1)
    for section_label, groups in SECTIONS_SPLIT:
        doc.add_heading(section_label, level=2)
        for _, _, metrics, subtags in groups[:1]:  # Aggregated is same for all groups, use first one
            for metric_label, filename_prefix in metrics:
                doc.add_heading(metric_label, level=3)
                for subtag in subtags:
                    doc.add_paragraph(f"{subtag.capitalize()} Trajectory", style="Heading 4")
                    svg_path = os.path.join(EVAL_PATH, f"{filename_prefix}_{subtag}.svg")
                    png_path = svg_path.replace(".svg", ".png")
                    if os.path.exists(svg_path):
                        convert_svg_if_needed(svg_path, png_path)
                        run = doc.add_paragraph().add_run()
                        run.add_picture(png_path, width=Inches(5.5))
                    else:
                        doc.add_paragraph(f"[Missing: {svg_path}]")

def generate_base_model_summary():
    doc = Document()
    doc.add_heading("Base Model Summary Report", 0)

    for section_title, filename, subdir in SECTIONS:
        doc.add_heading(section_title, level=1)
        image_paths = []

        for topic in TOPICS:
            if subdir is None:
                svg_path = os.path.join(EVAL_PATH, topic, filename)
            else:
                svg_path = os.path.join(EVAL_PATH, topic, subdir, filename)

            png_path = svg_path.replace(".svg", ".png")

            if os.path.exists(svg_path):
                try:
                    convert_svg_if_needed(svg_path, png_path)
                    image_paths.append(png_path)
                except Exception as e:
                    print(f"[Error converting image for {topic}]: {str(e)}")
            else:
                print(f"[Missing] {svg_path}")

        if image_paths:
            insert_images_in_rows(doc, image_paths)

        # Insert aggregated "All" image if in Human vs. LLM section
        if section_title.startswith("Human vs. LLM"):
            if "Bias" in section_title:
                if "Slider" in section_title:
                    agg_svg = os.path.join(EVAL_PATH, "summary_bias_errorbar_slider.svg")
            elif "Stance" in section_title:
                agg_svg = os.path.join(EVAL_PATH, "summary_bias_errorbar_stance.svg")
            else:   
                agg_svg = os.path.join(EVAL_PATH, "summary_bias_errorbar.svg")
        elif "Diversity" in section_title:
            if "Slider" in section_title:
                agg_svg = os.path.join(EVAL_PATH, "summary_std_errorbar_slider.svg")
            elif "Stance" in section_title:
                agg_svg = os.path.join(EVAL_PATH, "summary_std_errorbar_stance.svg")
            else:
                agg_svg = os.path.join(EVAL_PATH, "summary_std_errorbar.svg")
        else:
            agg_svg = None


            if agg_svg and os.path.exists(agg_svg):
                agg_png = agg_svg.replace(".svg", ".png")
                try:
                    convert_svg_if_needed(agg_svg, agg_png)
                    doc.add_paragraph("Aggregated Summary Across All Topics:")
                    run = doc.add_paragraph().add_run()
                    run.add_picture(agg_png, width=Inches(5.5))  # Full-width inline
                except Exception as e:
                    print(f"[Error inserting aggregated image {agg_svg}]: {e}")
            else:
                print(f"[Missing aggregated image] {agg_svg}")

    os.makedirs(EVAL_PATH, exist_ok=True)
    generate_transition_matrix_section(doc)
    generate_aggregated_transition_section(doc)
    append_metadata(doc)
    doc.save(OUTPUT_PATH)
    print(f"✅ Base model summary saved to: {OUTPUT_PATH}")

def generate_base_model_summary_publication():
    doc = Document()
    doc.add_heading("Base Model Summary Report (Publication Version)", 0)

    for section_title, filename, subdir in SECTIONS_PUBLICATION:
        doc.add_heading(section_title, level=1)
        image_paths = []

        for topic in TOPICS:
            if subdir is None:
                svg_path = os.path.join(EVAL_PATH, topic, filename)
            else:
                svg_path = os.path.join(EVAL_PATH, topic, subdir, filename)

            png_path = svg_path.replace(".svg", ".png")

            if os.path.exists(svg_path):
                try:
                    convert_svg_if_needed(svg_path, png_path)
                    image_paths.append(png_path)
                except Exception as e:
                    print(f"[Error converting image for {topic}]: {str(e)}")
            else:
                print(f"[Missing] {svg_path}")

        if image_paths:
            insert_images_in_rows(doc, image_paths)

        # Aggregated image (updated logic reused from your earlier block)
        if section_title.startswith("Human vs. LLM"):
            if "Bias" in section_title:
                if "Slider" in section_title:
                    agg_svg = os.path.join(EVAL_PATH, "summary_bias_errorbar_slider_publication.svg")
                elif "Stance" in section_title:
                    agg_svg = os.path.join(EVAL_PATH, "summary_bias_errorbar_stance_publication.svg")
                else:
                    agg_svg = os.path.join(EVAL_PATH, "summary_bias_errorbar_publication.svg")
            elif "Diversity" in section_title:
                if "Slider" in section_title:
                    agg_svg = os.path.join(EVAL_PATH, "summary_std_errorbar_slider_publication.svg")
                elif "Stance" in section_title:
                    agg_svg = os.path.join(EVAL_PATH, "summary_std_errorbar_stance_publication.svg")
                else:
                    agg_svg = os.path.join(EVAL_PATH, "summary_std_errorbar_publication.svg")
            else:
                agg_svg = None

            if agg_svg and os.path.exists(agg_svg):
                agg_png = agg_svg.replace(".svg", ".png")
                try:
                    convert_svg_if_needed(agg_svg, agg_png)
                    doc.add_paragraph("Aggregated Summary Across All Topics:")
                    run = doc.add_paragraph().add_run()
                    run.add_picture(agg_png, width=Inches(5.5))
                except Exception as e:
                    print(f"[Error inserting aggregated image {agg_svg}]: {e}")
            else:
                print(f"[Missing aggregated image] {agg_svg}")

    generate_transition_matrix_section(doc)
    generate_aggregated_transition_section(doc)
    append_metadata(doc)

    doc.save(OUTPUT_PATH_PUBLICATION)
    print(f"✅ Base model publication summary saved to: {OUTPUT_PATH_PUBLICATION}")

def generate_split_errorbar_publication_docx():
    doc = Document()
    doc.add_heading("Base Model Summary (Split View – Publication Version)", 0)

    def insert_grid_images(image_paths):
        if image_paths:
            insert_images_in_rows(doc, image_paths, images_per_row=3)
        else:
            doc.add_paragraph("[Missing images]")

    for section_label, groups in SECTIONS_SPLIT:
        doc.add_page_break()
        doc.add_heading(section_label, level=1)

        for group_label, source, metrics, subtags in groups:
            doc.add_heading(f"{group_label}:", level=2)
            for metric_label, filename_prefix in metrics:
                doc.add_heading(metric_label, level=3)
                for subtag in subtags:
                    doc.add_paragraph(f"{subtag.capitalize()} Trajectory", style="Heading 4")
                    image_paths = []
                    for topic in TOPICS:
                        if source is None:
                            svg_path = os.path.join(EVAL_PATH, topic, f"{filename_prefix}_{subtag}_publication.svg")
                        else:
                            svg_path = os.path.join(EVAL_PATH, topic, source, f"{filename_prefix}_{subtag}_publication.svg")
                        png_path = svg_path.replace(".svg", ".png")
                        if os.path.exists(svg_path):
                            convert_svg_if_needed(svg_path, png_path)
                            image_paths.append(png_path)
                    insert_grid_images(image_paths)

    doc.add_page_break()
    doc.add_heading("Aggregated Plots", level=1)
    for section_label, groups in SECTIONS_SPLIT:
        doc.add_heading(section_label, level=2)
        for _, _, metrics, subtags in groups[:1]:
            for metric_label, filename_prefix in metrics:
                doc.add_heading(metric_label, level=3)
                for subtag in subtags:
                    doc.add_paragraph(f"{subtag.capitalize()} Trajectory", style="Heading 4")
                    svg_path = os.path.join(EVAL_PATH, f"{filename_prefix}_{subtag}_publication.svg")
                    png_path = svg_path.replace(".svg", ".png")
                    if os.path.exists(svg_path):
                        convert_svg_if_needed(svg_path, png_path)
                        run = doc.add_paragraph().add_run()
                        run.add_picture(png_path, width=Inches(5.5))
                    else:
                        doc.add_paragraph(f"[Missing: {svg_path}]")

    output_path = os.path.join(EVAL_PATH, "base_model_summary_split_publication.docx")
    os.makedirs(EVAL_PATH, exist_ok=True)
    doc.save(output_path)
    print(f"✅ Saved publication summary to: {output_path}")

def generate_split_errorbar_summary():
    doc = Document()
    doc.add_heading("Base Model Summary (Split View)", 0)
    generate_split_errorbar_section(doc)
    os.makedirs(EVAL_PATH, exist_ok=True)
    doc.save(os.path.join(EVAL_PATH, "base_model_summary_split.docx"))
    print("✅ Saved: base_model_summary_split.docx")

if __name__ == "__main__":
    generate_base_model_summary()
    generate_base_model_summary_publication()
    generate_split_errorbar_summary()
    generate_split_errorbar_publication_docx()
