"""
Create DOCX summaries for grouped bar-histograms with the layout:
1) Slider:   Human | LLM | Δ
2) Stance:   Human | LLM | Δ
3) Tweet:    Human | LLM | Δ

Outputs (under histograms folder):
- histogram_summary.docx         (regular SVGs)
- histogram_summary_pub.docx     (publication SVGs: *_pub.svg)
"""

import os
from io import BytesIO
from docx import Document
from docx.shared import Inches
from docx.enum.text import WD_ALIGN_PARAGRAPH
import ctypes
import ctypes.util

# -------------------
# Cairo (Homebrew) loader for macOS and Linux
# -------------------
def resolve_cairo_path():
    # 1) Try the system’s library resolver (portable)
    lib = ctypes.util.find_library("cairo")
    if lib:
        return lib  # e.g., "libcairo.so.2" or absolute path

    # 2) Try common absolute paths (Linux)
    candidates = [
        "/lib64/libcairo.so.2",
        "/usr/lib64/libcairo.so.2",
        "/usr/lib/x86_64-linux-gnu/libcairo.so.2",
        "/lib64/libcairo.so",
        "/lib/x86_64-linux-gnu/libcairo.so.2",
    ]
    for p in candidates:
        if os.path.exists(p):
            return p

    # 3) Optional: allow override via env var
    env_path = os.environ.get("LIBCAIRO_PATH")
    if env_path and os.path.exists(env_path):
        return env_path

    return None

lib_path = resolve_cairo_path()
if not lib_path:
    raise FileNotFoundError(
        "Cairo library not found. Install libcairo (e.g., apt/yum), or set LIBCAIRO_PATH to the .so path."
    )

# Load it so downstream libs (e.g., cairosvg) can find symbols
ctypes.CDLL(lib_path)
print(f"[cairo] Using {lib_path}")

import cairosvg

# -------------------
# Paths
# -------------------
# -------------------
# Paths
# -------------------
PROJECT_ROOT = os.path.abspath(os.path.join(os.path.dirname(__file__), "..", ".."))
IMG_DIR = os.path.join(PROJECT_ROOT, "result", "group_level_eval", "histograms")
os.makedirs(IMG_DIR, exist_ok=True)

OUT_DOCX_REGULAR = os.path.join(IMG_DIR, "histogram_summary.docx")
OUT_DOCX_PUB     = os.path.join(IMG_DIR, "histogram_summary_pub.docx")

# Publication docx: set to False to remove captions entirely
ADD_CAPTIONS_IN_PUB = True

# -------------------
# Grouped image manifests
# -------------------
# Each group: (Group Title, [(caption, filename), (caption, filename), (caption, filename)])
GROUPS_REGULAR = [
    (
        "Group 1 — Slider (Initial vs Post)",
        [
            ("Human: Initial vs Post (slider)", "barhist_human_initial_vs_post_slider.svg"),
            ("LLM:   Initial vs Post (slider)", "barhist_llm_initial_vs_post_slider.svg"),
            ("Δ (slider): Post − Initial — Human vs LLM", "barhist_diff_post_minus_initial_slider.svg"),
        ],
    ),
    (
        "Group 2 — Stance (Initial vs Post)",
        [
            ("Human: Initial vs Post (stance)", "barhist_human_initial_vs_post_stance.svg"),
            ("LLM:   Initial vs Post (stance)", "barhist_llm_initial_vs_post_stance.svg"),
            ("Δ (stance): Post − Initial — Human vs LLM", "barhist_diff_post_minus_initial_stance.svg"),
        ],
    ),
    (
        "Group 3 — Tweet (Tweet 1 vs Tweet 3)",
        [
            ("Human: Tweet 1 vs Tweet 3", "barhist_human_tweet1_vs_tweet3.svg"),
            ("LLM:   Tweet 1 vs Tweet 3", "barhist_llm_tweet1_vs_tweet3.svg"),
            # Δ uses the same file you already generate (named *stance*)
            ("Δ (tweet): Tweet 3 − Tweet 1 — Human vs LLM", "barhist_diff_tweet3_minus_tweet1_stance.svg"),
        ],
    ),
]

GROUPS_PUB = [
    (
        "Group 1 — Slider (Initial vs Post)",
        [
            ("Human: Initial vs Post (slider)", "barhist_human_initial_vs_post_slider_pub.svg"),
            ("LLM:   Initial vs Post (slider)", "barhist_llm_initial_vs_post_slider_pub.svg"),
            ("Δ (slider): Post − Initial — Human vs LLM", "barhist_diff_post_minus_initial_slider_pub.svg"),
        ],
    ),
    (
        "Group 2 — Stance (Initial vs Post)",
        [
            ("Human: Initial vs Post (stance)", "barhist_human_initial_vs_post_stance_pub.svg"),
            ("LLM:   Initial vs Post (stance)", "barhist_llm_initial_vs_post_stance_pub.svg"),
            ("Δ (stance): Post − Initial — Human vs LLM", "barhist_diff_post_minus_initial_stance_pub.svg"),
        ],
    ),
    (
        "Group 3 — Tweet (Tweet 1 vs Tweet 3)",
        [
            ("Human: Tweet 1 vs Tweet 3", "barhist_human_tweet1_vs_tweet3_pub.svg"),
            ("LLM:   Tweet 1 vs Tweet 3", "barhist_llm_tweet1_vs_tweet3_pub.svg"),
            ("Δ (tweet): Tweet 3 − Tweet 1 — Human vs LLM", "barhist_diff_tweet3_minus_tweet1_stance_pub.svg"),
        ],
    ),
]

# -------------------
# Helpers
# -------------------
def svg_to_png_bytes(svg_path: str) -> BytesIO:
    """Convert an SVG file to PNG bytes (for docx embedding)."""
    png_bytes = cairosvg.svg2png(url=svg_path)
    bio = BytesIO(png_bytes)
    bio.seek(0)
    return bio

def add_image_with_caption(cell, png_bytesio, caption: str, width_in=2.35, add_caption=True):
    """
    Insert an image (PNG in memory) into a docx table cell with an optional caption below.
    3 columns per row => ~2.3–2.4 inches wide fits well on US Letter with margins.
    """
    if not cell.paragraphs:
        cell.add_paragraph()
    run = cell.paragraphs[0].add_run()
    run.add_picture(png_bytesio, width=Inches(width_in))
    cell.paragraphs[0].alignment = WD_ALIGN_PARAGRAPH.CENTER

    if add_caption:
        p = cell.add_paragraph(caption)
        p.alignment = WD_ALIGN_PARAGRAPH.CENTER

def add_group_table(doc: Document, title: str, items, add_captions=True):
    """
    Add a group heading and a 1x3 table (Human | LLM | Δ).
    'items' is a list of 3 (caption, filename).
    """
    # Group heading
    h = doc.add_heading(title, level=2)
    h.alignment = WD_ALIGN_PARAGRAPH.LEFT

    # 1 row x 3 columns
    table = doc.add_table(rows=1, cols=3)
    table.autofit = True

    for i, (caption, fname) in enumerate(items):
        cell = table.cell(0, i)
        svg_path = os.path.join(IMG_DIR, fname)
        if not os.path.exists(svg_path):
            cell.text = f"[Missing] {caption}\n{svg_path}"
            continue
        png_io = svg_to_png_bytes(svg_path)
        add_image_with_caption(cell, png_io, caption, width_in=2.35, add_caption=add_captions)

def build_docx(group_manifest, out_path, title, add_captions=True):
    doc = Document()
    doc.add_heading(title, level=1)

    for group_title, items in group_manifest:
        add_group_table(doc, group_title, items, add_captions=add_captions)

    doc.save(out_path)
    print(f"✅ DOCX saved: {out_path}")

# -------------------
# Build both docs
# -------------------
if __name__ == "__main__":
    # Regular (with captions)
    build_docx(
        GROUPS_REGULAR,
        OUT_DOCX_REGULAR,
        title="Grouped Bar Histograms — Slider / Stance / Tweet (Human | LLM | Δ)",
        add_captions=True
    )

    # Publication (uses *_pub.svg; captions optional)
    build_docx(
        GROUPS_PUB,
        OUT_DOCX_PUB,
        title="Grouped Bar Histograms (Publication) — Slider / Stance / Tweet (Human | LLM | Δ)",
        add_captions=ADD_CAPTIONS_IN_PUB
    )