import os
import glob
from docx import Document
from docx.shared import Inches
from docx.enum.text import WD_ALIGN_PARAGRAPH

import ctypes
import ctypes.util

def resolve_cairo_path():
    # 1) Try the system’s library resolver (portable)
    lib = ctypes.util.find_library("cairo")
    if lib:
        return lib  # e.g., "libcairo.so.2" or absolute path

    # 2) Try common absolute paths (Linux)
    candidates = [
        "/lib64/libcairo.so.2",
        "/usr/lib64/libcairo.so.2",
        "/usr/lib/x86_64-linux-gnu/libcairo.so.2",
        "/lib64/libcairo.so",
        "/lib/x86_64-linux-gnu/libcairo.so.2",
    ]
    for p in candidates:
        if os.path.exists(p):
            return p

    # 3) Optional: allow override via env var
    env_path = os.environ.get("LIBCAIRO_PATH")
    if env_path and os.path.exists(env_path):
        return env_path

    return None

lib_path = resolve_cairo_path()
if not lib_path:
    raise FileNotFoundError(
        "Cairo library not found. Install libcairo (e.g., apt/yum), or set LIBCAIRO_PATH to the .so path."
    )

# Load it so downstream libs (e.g., cairosvg) can find symbols
ctypes.CDLL(lib_path)
print(f"[cairo] Using {lib_path}")

import cairosvg

# -------------------
# Paths
# -------------------
PROJECT_ROOT     = os.path.abspath(os.path.join(os.path.dirname(__file__), "..", ".."))
PLOT_OUTPUT_ROOT = os.path.join(PROJECT_ROOT, "result", "group_level_eval", "plots")
os.makedirs(PLOT_OUTPUT_ROOT, exist_ok=True)

OUT_REGULAR = os.path.join(PLOT_OUTPUT_ROOT, "scatter_grid_summary.docx")
OUT_PUB     = os.path.join(PLOT_OUTPUT_ROOT, "scatter_grid_summary_pub.docx")

# Topic to hide (we only want the *_reversed pseudo-topic in summaries)
REVERSE_TOPIC_RAW = "Everything_that_happens_can_eventually_be_explained_by_science"

# === PLOT FILE KEYS (FILENAMES MUST MATCH) ===
PLOT_KEYS = {
    # ---- Group 1: Stance • Tweet agreement & predictors ----
    "stance_scatter_tweet3_vs_tweet1":                             "Tweet 3 vs Tweet 1 (Stance)",
    "stance_scatter_tweet3_minus_tweet1_vs_tweet1":                "Δ Stance vs Tweet 1",
    "stance_scatter_tweet3_minus_tweet1_vs_abs_avg_partner12_minus_tweet1abs_": "Δ Stance vs |avg_partner12 − Tweet1|",
    "stance_scatter_tweet3_minus_tweet1_vs_avg_partner12_minus_tweet1":     "Δ Stance vs (avg_partner12 − Tweet1)",
    "stance_scatter_tweet3_minus_tweet1_vs_abs_partner_minus_tweet1abs_":       "Δ Stance vs |partner − Tweet1|",
    "stance_scatter_tweet3_minus_tweet1_vs_partner_minus_tweet1":           "Δ Stance vs (partner − Tweet1)",
    "stance_scatter_tweet3_minus_tweet1_vs_partner2_t2_minus_tweet1":       "Δ Stance vs (partner2 T2 − Tweet1)",
    "stance_scatter_tweet3_minus_tweet1_vs_partner3_t3_minus_tweet1":       "Δ Stance vs (partner3 T3 − Tweet1)",

    # ---- Group 2: Stance • Opinion (Initial ↔ Post) ----
    "stance_scatter_post_vs_initial":                               "Post Stance vs Initial Stance",
    "stance_scatter_post_minus_initial_vs_initial":                 "Δ Stance vs Initial Stance",
    "stance_scatter_post_minus_initial_vs_avg_others_minus_initial":"Δ Stance vs (avg others − Initial Stance)",
    "stance_scatter_post_minus_initial_vs_partner_minus_initial":   "Δ Stance vs (partner − Initial Stance)",
    "stance_scatter_post_minus_initial_vs_abs_partner_minus_initialabs_":"Δ Stance vs |partner − Initial| Stance",

    # ---- Group 3: Slider • Initial ↔ Post ----
    "slider_scatter_post_vs_initial":                                     "Post Slider vs Initial Slider",
    "slider_scatter_post_minus_initial_vs_initial":                       "Δ Slider vs Initial Slider",
    "slider_scatter_post_minus_initial_vs_avg_others_minus_initial":      "Δ Slider vs (avg others − Initial Slider)",
    "slider_scatter_post_minus_initial_vs_partner_minus_initial":         "Δ Slider vs (partner − Initial Slider)",
    "slider_scatter_post_minus_initial_vs_abs_partner_minus_initialabs_": "Δ Slider vs |partner − Initial| Slider",

    # ---- Group 4: Paired Δ (LLM vs Human; y vs x) ----
    "paired_delta_slider": "Δ Slider (LLM vs Human; y vs x)",
    "paired_delta_stance": "Δ Stance (Post−Initial) (LLM vs Human; y vs x)",
    "paired_delta_tweet":  "Δ Tweet (T3−T1) (LLM vs Human; y vs x)",

}

# -------------------
# Helpers
# -------------------
def convert_svg_to_png(svg_path: str, png_path: str):
    png_bytes = cairosvg.svg2png(url=svg_path)
    with open(png_path, "wb") as f:
        f.write(png_bytes)

def insert_images_as_table_grid(doc: Document, image_paths, cols=3, width_in=2.25):
    """Insert up to 9 images as a 3×3 grid. None entries leave cells blank."""
    rows = (len(image_paths) + cols - 1) // cols
    table = doc.add_table(rows=rows, cols=cols)
    table.autofit = True
    for idx, img in enumerate(image_paths):
        r, c = divmod(idx, cols)
        cell = table.cell(r, c)
        if img and os.path.exists(img):
            if not cell.paragraphs:
                cell.add_paragraph()
            run = cell.paragraphs[0].add_run()
            run.add_picture(img, width=Inches(width_in))
            cell.paragraphs[0].alignment = WD_ALIGN_PARAGRAPH.CENTER
        else:
            # optional: placeholder text
            # cell.text = ""
            pass

def collect_topic_dirs():
    """Return sorted list of topic directories, excluding the raw reverse topic name."""
    dirs = []
    for topic_dir in sorted(glob.glob(os.path.join(PLOT_OUTPUT_ROOT, "*"))):
        if not os.path.isdir(topic_dir):
            continue
        topic = os.path.basename(topic_dir)
        if topic == REVERSE_TOPIC_RAW:
            continue
        dirs.append(topic_dir)
    return dirs

def build_grid_for_key(doc: Document, key_base: str, title: str, publication: bool):
    doc.add_heading(title, level=1)

    # Build list: 7 per-topic + 1 aggregated + 1 blank = 9 slots
    image_paths = []
    topic_dirs = collect_topic_dirs()

    # Pick first 7 topics (adjust if you prefer a specific ordering)
    for topic_dir in topic_dirs[:7]:
        svg = os.path.join(topic_dir, f"{key_base}{'_publication' if publication else ''}.svg")
        png = svg.replace(".svg", "_converted.png")
        if os.path.exists(svg):
            convert_svg_to_png(svg, png)
            image_paths.append(png)

    # Aggregated (root) as the 8th
    agg_svg = os.path.join(PLOT_OUTPUT_ROOT, f"{key_base}{'_publication' if publication else ''}.svg")
    agg_png = agg_svg.replace(".svg", "_converted.png")
    if os.path.exists(agg_svg):
        convert_svg_to_png(agg_svg, agg_png)
        image_paths.append(agg_png)

    # Pad to 9
    while len(image_paths) < 9:
        image_paths.append(None)

    insert_images_as_table_grid(doc, image_paths)

def generate_docx(publication: bool, out_path: str):
    doc = Document()
    doc.add_heading("Scatter Plot Summary", 0)
    doc.add_paragraph("Human vs. LLM (gpt-4o-mini-2024-07-18)",
                      style="Intense Quote")

    for key_base, title in PLOT_KEYS.items():
        build_grid_for_key(doc, key_base, title, publication)

    doc.save(out_path)
    print(f"✅ Saved summary report: {out_path}")

# -------------------
# Build both (regular + publication)
# -------------------
if __name__ == "__main__":
    generate_docx(publication=False, out_path=OUT_REGULAR)
    generate_docx(publication=True,  out_path=OUT_PUB)