import os
from docx import Document
from docx.shared import Inches
import cairosvg

# Set DYLD_FALLBACK_LIBRARY_PATH before running if needed:
# export DYLD_FALLBACK_LIBRARY_PATH=/opt/homebrew/opt/cairo/lib

PROJECT_ROOT = os.path.abspath(os.path.join(os.path.dirname(__file__), "..", ".."))
RESULT_DIR = os.path.join(PROJECT_ROOT, "result", "group_level_eval")
OUTPUT_PATH = os.path.join(RESULT_DIR, "split_analysis_summary.docx")

SPLITS = ["group_split", "topic_split", "round_split"]
TYPES = ["train", "test"]
METRICS = {
    "opinion": "Opinion Diversity (Post - Initial)",
    "stance": "Stance Diversity (Tweet3 - Tweet1)"
}

def svg_to_png(svg_path):
    png_path = svg_path.replace(".svg", ".png")
    if not os.path.exists(png_path) or os.path.getmtime(png_path) < os.path.getmtime(svg_path):
        cairosvg.svg2png(url=svg_path, write_to=png_path)
    return png_path

def add_chart(doc, title, svg_path):
    if not os.path.exists(svg_path):
        print(f"[SKIP] Missing: {svg_path}")
        return
    png_path = svg_to_png(svg_path)
    doc.add_heading(title, level=2)
    doc.add_picture(png_path, width=Inches(6))
    doc.add_paragraph("")

def generate_summary():
    doc = Document()
    doc.add_heading("Group-Level Diversity Analysis Summary", level=1)

    for split in SPLITS:
        title = split.replace("_split", "").capitalize() + " Split"
        doc.add_heading(title, level=1)

        for dtype in TYPES:
            subtitle = "Training" if dtype == "train" else "Testing"
            doc.add_heading(subtitle, level=2)

            for metric_key, metric_title in METRICS.items():
                fname = f"{dtype}_{split}_{metric_key}_diversity.svg"
                fpath = os.path.join(RESULT_DIR, f"{split}_depth", fname)
                title = f"{metric_title}"
                add_chart(doc, title, fpath)

    doc.save(OUTPUT_PATH)
    print(f"[INFO] Summary saved to: {OUTPUT_PATH}")

if __name__ == "__main__":
    generate_summary()
