#!/usr/bin/env python3
import argparse
import os
import subprocess
import sys
import time

STEPS = [
    ["preprocessing.py", "--mode", "depth"],
    ["base_model_analysis.py"],
    ["paired_ttest.py"],
    ["plot_base_stats.py"],
    ["plot_base_transition_mat.py"],
    ["generate_base_model_summary.py"],
    ["draw_scatter_plots.py"],
    ["generate_scatter_plot_summary.py"],
    ["draw_histogram.py"],
    ["generate_histogram_summary.py"],
]

def run(cmd, cwd, dry_run=False):
    pretty = " ".join([cmd[0]] + [sub if " " not in sub else f'"{sub}"' for sub in cmd[1:]])
    print(f"\n▶️  Running: {pretty}")
    if dry_run:
        print("   (dry-run) skipped execution")
        return 0, 0.0
    start = time.time()
    proc = subprocess.run([sys.executable, *cmd], cwd=cwd)
    elapsed = time.time() - start
    print(f"   ⏱  Finished in {elapsed:.2f}s with code {proc.returncode}")
    return proc.returncode, elapsed

def main():
    parser = argparse.ArgumentParser(description="Run group-level evaluation pipeline end-to-end.")
    parser.add_argument("--continue-on-error", action="store_true",
                        help="Continue running remaining steps even if a step fails.")
    parser.add_argument("--dry-run", action="store_true",
                        help="Print steps without executing them.")
    parser.add_argument("--from-step", type=int, default=1,
                        help="1-based index of the step to start from (default: 1).")
    parser.add_argument("--to-step", type=int, default=len(STEPS),
                        help=f"1-based index of the step to end at (default: {len(STEPS)}).")
    args = parser.parse_args()

    # Run everything relative to this script's directory
    base_dir = os.path.abspath(os.path.dirname(__file__))

    total_start = time.time()
    failures = []
    from_idx = max(1, args.from_step)
    to_idx = min(len(STEPS), args.to_step)

    print("🧪 Pipeline steps:")
    for i, step in enumerate(STEPS, start=1):
        print(f"  {i}. {' '.join(step)}")
    print(f"\nSelected range: steps {from_idx} → {to_idx}")
    if args.dry_run:
        print("Mode: dry-run (no commands will execute)")

    for i in range(from_idx, to_idx + 1):
        step_cmd = STEPS[i - 1]
        code, _ = run(step_cmd, cwd=base_dir, dry_run=args.dry_run)
        if code != 0:
            failures.append((i, " ".join(step_cmd), code))
            print(f"❌ Step {i} failed with exit code {code}.")
            if not args.continue_on_error:
                print("Stopping due to failure. Use --continue-on-error to run remaining steps.")
                break

    total_elapsed = time.time() - total_start
    print("\n====================")
    print(f"🏁 Pipeline finished in {total_elapsed:.2f}s")
    if failures:
        print("⚠️  Failures:")
        for idx, cmd, code in failures:
            print(f"   - Step {idx}: {cmd} (exit {code})")
        sys.exit(1)
    else:
        print("✅ All selected steps completed successfully.")
        sys.exit(0)

if __name__ == "__main__":
    main()
